// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief for 14,22,34.8 and 43.5 GeV
  class TASSO_1987_I248660 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1987_I248660);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en], ih, 1, 1);
        book(_c[en], "TMP/weightSum"+en);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const FinalState& fs = apply<FinalState>(event, "FS");
      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if ( fs.particles().size() < 2)  vetoEvent;

      _c[_sqs]->fill();
      const double Evis2 = sqr(sum(fs.particles(), Kin::E, 0.0));
      // (A)EEC
      // Need iterators since second loop starts at current outer loop iterator, i.e. no "foreach" here!
      for (Particles::const_iterator p_i = fs.particles().begin(); p_i != fs.particles().end(); ++p_i) {
        for (Particles::const_iterator p_j = p_i; p_j != fs.particles().end(); ++p_j) {
          const Vector3 mom3_i = p_i->p3();
          const Vector3 mom3_j = p_j->p3();
          const double energy_i = p_i->E();
          const double energy_j = p_j->E();
          const double cosij = dot(mom3_i.unit(), mom3_j.unit());
          double eec = (energy_i*energy_j) / Evis2;
          if (p_i != p_j) eec *= 2.0;
          _h[_sqs]->fill(cosij, eec);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (const auto& item : _c) {
        scale(_h[item.first],  1.0/item.second->sumW());
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    map<string,CounterPtr> _c;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1987_I248660);
}
