# Correlated Subevent Handling

Rivet supports the analysis of NLO fixed-order events with correlated subevents,
i.e. sets of `HepMC::GenEvent` records that share the same event number and
 correspond to weighted contributions from a single physical event.

Correlated subevents arise in fixed-order multi-weight generators and must be
collapsed into a single fill to correctly account for partial cancellations
 and avoid double-counting in histograms and counters.

Rivet automatically detects and caches subevents by event number,
 and processes the full group only once all correlated subevents are known.

---

## Unbinned Counters

For counters (e.g. total cross-section estimates), correlated subevents are
collapsed via a sum over their fractional weights per analysis weight.
In pseudocode, the logic looks something like this:
```cpp
for (weight : multiweights) {
  sumfw = [0.0] * nFills;
  for (grp : eventGroup) {
    for (i, fill : enumerate(grp.fills)) {
      sumfw[i] += fill.fraction * weight;
    }
  }
  for (fw : sumfw) {
    persistent[weight_idx]->fill(fw);
  }
}
```


## Binned Distributions

For histogrammed observables, care must be taken when correlated subevents
fall into different bins. This leads to bin-edge migration effects, which,
if not handled properly, can distort uncertainties.

### Smearing and Fill Windows

To address this, Rivet uses fill windows: correlated subevents that fall into
nearby bins (within a smearing window) are grouped and collapsed. The windowing
logic supports arbitrary dimensions.

Each fill window represents a single persistent histogram entry, summarising:
- The collapsed fill coordinate
-	The effective fill weight, optionally normalised
-	The fill fraction, encoding the number of subevents used

The windowing loop proceeds by:
1. Computing the maximum fill count across all subevents
2. Iterating backwards from the maximum to ensure all fills are used
3. Within each fill index, checking overlap between subevents to define a subwindow
4. Applying either:
   -	Collapsed weight from all subevents (no smearing), or
   - Window-local weights, with optional normalisation (with smearing)

With smearing and bin-edge migration, multiple fill windows may be formed,
potentially inflating the sum of squared weights, leading to overestimated uncertainties
 — especially for infrared-unsafe observables (e.g. parton multiplicities).

### Example: Misaligned Subevents

Consider three correlated subevents with weights `{2.0, 3.0, -4.0}`.
Suppose two land in the same histogram bin and one in a different bin.
With smearing enabled, the subevents are split into two fill windows:
```cpp
bin1.fill(+6.0, 1/3);   // from weight 2.0, frac = 1/3
bin2.fill(-1.5, 2/3);   // from weights 3.0 and -4.0, frac = 2/3
```
(Note that the weights are normalised to a single fill with `weight/fillFrac`.)
Now suppose we rebin into a single bin.
The central value (sum of weights) is preserved:
```py
sumW  = (6.0 * 1/3) + (-1.5 * 2/3) = 2 - 1 = 1
```
But the statistical uncertainty is inflated:
```py
sumW2 = (6.0)**2 * 1/3 + (-1.5)**2 * 2/3 = 12 + 1.5 = 13.5
```
By contrast, collapsing the subevents before filling the histogram would yield:
```py
sumW  = 2.0 + 3.0 - 4.0 = 1
sumW2 = (2 + 3 - 4)**2   = 1
```

### Usage notes

-	Smearing can be controlled via the `fsmear` parameter (e.g. using `--nlo-smearing 0.5`).
- The default behaviour is to not apply the smearing.
-	Partial fill collapse is applied automatically if subevents fall into the same bin.
-	Uncertainty inflation can occur in discrete observables when partial cancellations are lost due to bin migration.
-	Rivet emits a runtime warning when correlated subevents are detected, encouraging users to check the robustness of their binning scheme.

