#!/usr/bin/env bash
source ../common.sh

./clean.sh

# ---
# This test covers Lake's interactions with the Lean module system.
# ---

mkdir Test/Generated
cat > Test/Generated/Module.lean <<EOF
module

-- insert here

/-- docstring A -/
public def foo : String := "bar"
EOF

# Test cross-package `import all``
test_err "cannot 'import all' across packages" build ErrorTest.CrossPackageImportAll

# Tests importing of a module's private segment
# should not not be imported by a plain `import` in a module
test_run build Test.Module.Import
test_cmd_fails grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/Import.setup.json
test_run build Test.Module.PublicImport
test_cmd_fails grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/PublicImport.setup.json
# should be imported by an `import all` in a module
test_run build Test.Module.ImportAll
test_cmd grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/ImportAll.setup.json
# including promoted imports
test_run build Test.Module.PromoteImport
test_cmd grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/PromoteImport.setup.json
test_run build Test.Module.PromoteTransImport
test_cmd grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/PromoteTransImport.setup.json
# should be imported by a non-module
test_run build Test.NonModule.Import
test_cmd grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/NonModule/Import.setup.json
# should not be imported by a module transitive import of a private `import all`
test_run build Test.Module.ImportImportAll
test_cmd_fails grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/ImportImportAll.setup.json
# should not be imported by an `import all` of a private module `import`
test_run build Test.Module.ImportAllImport
test_cmd_fails grep -F "Test/Generated/Module.olean.private" .lake/build/ir/Test/Module/ImportAllImport.setup.json

# Tests that the transitive module import of a private import does not include its artifacts
test_run build Test.Module.ImportImport
test_cmd_fails grep -F "Test/Generated/Module.olean" .lake/build/ir/Test/Module/ImportImport.setup.json

# Build all tests before making an edit
test_run build

# Tests a public edit of an import
echo "# TEST: public edit"
test_cmd sed_i 's/foo/fu/' Test/Generated/Module.lean
old_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.hash)
test_out "Built Test.Generated.Module" build Test.Generated.Module -v
new_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.hash)
test_exp $old_hash != $new_hash
# should trigger a rebuild on a module direct `import`
test_out "Built Test.Module.Import" build Test.Module.Import -v
test_out "Built Test.Module.PublicImport" build Test.Module.PublicImport -v
# should trigger a rebuild on a direct `import all`
test_out "Built Test.Module.ImportAll" build Test.Module.ImportAll -v
# should trigger a rebuild for a non-module direct import
test_out "Built Test.NonModule.Import" build Test.NonModule.Import -v
# should not trigger a rebuild on a module transitive private `import`
test_run build Test.Module.ImportImport --no-build
# should trigger a rebuild on an `import all` of a transitive private `import`
test_out "Built Test.Module.ImportAllImport" build Test.Module.ImportAllImport -v
# should trigger a rebuild for a non-module transitive import
test_out "Built Test.NonModule.ImportModuleImport" build Test.NonModule.ImportModuleImport -v

# Build all tests before making an edit
test_run build

# Tests a private edit of an import
echo "# TEST: private edit"
test_cmd sed_i 's/bar/baz/' Test/Generated/Module.lean
old_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.private.hash)
old_pub_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.hash)
test_out "Built Test.Generated.Module" build Test.Generated.Module -v
new_pub_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.hash)
new_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.private.hash)
test_exp $old_pub_hash == $new_pub_hash
test_exp $old_hash != $new_hash
# should not trigger a rebuild on a module direct `import`
test_run build Test.Module.Import Test.Module.PublicImport --no-build
# should trigger a rebuild on a direct `import all`
test_out "Built Test.Module.ImportAll" build Test.Module.ImportAll -v
test_out "Built Test.Module.PromoteImport" build Test.Module.PromoteImport -v
# should trigger a rebuild for a non-module direct import
test_out "Built Test.NonModule.Import" build Test.NonModule.Import -v
# should not trigger a rebuild on an `import all` of a transitive private `import`
test_run build Test.Module.ImportAllImport --no-build
# should trigger a rebuild on an `import all` of a transitive private `import all`
test_out "Built Test.Module.ImportAllImportAll" build Test.Module.ImportAllImportAll -v
test_out "Built Test.Module.PromoteTransImport" build Test.Module.PromoteTransImport -v
# should trigger a rebuild for a non-module transitive import
test_out "Built Test.NonModule.ImportModuleImport" build Test.NonModule.ImportModuleImport -v

# Build all tests before making an edit
test_run build

# Tests a server edit of an import
echo "# TEST: server edit"
test_cmd sed_i '/-- insert here/{G;}' Test/Generated/Module.lean
old_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.server.hash)
test_out "Built Test.Generated.Module" build Test.Generated.Module -v
new_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.server.hash)
test_exp $old_hash != $new_hash
# should not trigger a rebuild on a direct module `import`
test_run build Test.Module.Import Test.Module.PublicImport --no-build
# should trigger a rebuild on a direct `import all`
test_out "Built Test.Module.ImportAll" build Test.Module.ImportAll -v
# should trigger a rebuild for a non-module direct import
test_out "Built Test.NonModule.Import" build Test.NonModule.Import -v
# should trigger a rebuild for a non-module transitive import
test_out "Built Test.NonModule.ImportModuleImport" build Test.NonModule.ImportModuleImport -v

# Build all tests before making an edit
test_run build

# Tests a meta edit of an import
echo "# TEST: meta edit"
test_cmd sed_i 's/baz/ipsum/' Test/Generated/Module.lean
old_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.ir.hash)
old_pub_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.hash)
test_out "Built Test.Generated.Module" build Test.Generated.Module -v
new_pub_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.olean.hash)
new_hash=$(cat .lake/build/lib/lean/Test/Generated/Module.ir.hash)
test_exp $old_pub_hash == $new_pub_hash
test_exp $old_hash != $new_hash
# should not trigger a rebuild on a direct module `import`
test_run build Test.Module.Import Test.Module.PublicImport --no-build
# should trigger a rebuild on a direct `meta import`
test_out "Built Test.Module.MetaImport" build Test.Module.MetaImport -v
test_out "Built Test.Module.PublicMetaImport" build Test.Module.PublicMetaImport -v
# should trigger a rebuild on a transitive `meta import`
test_out "Built Test.Module.MetaImportPublicImport" build Test.Module.MetaImportPublicImport -v
# should trigger a rebuild on module transitive import of a `public meta import`
test_out "Built Test.Module.ImportPublicMetaImport" build Test.Module.ImportPublicMetaImport -v
# should not trigger a rebuild on module transitive import of a private `meta import`
test_run build Test.Module.ImportMetaImport --no-build
# should trigger a rebuild on a direct `import all`
test_out "Built Test.Module.ImportAll" build Test.Module.ImportAll -v
# should not trigger a rebuild on an `import all` of a transitive private `import`
test_run build Test.Module.ImportAllImport --no-build
# should trigger a rebuild on a `import all` of a private `meta import`
test_out "Built Test.Module.ImportAllMetaImport" build Test.Module.ImportAllMetaImport -v
# should trigger a rebuild for a non-module direct import
test_out "Built Test.NonModule.Import" build Test.NonModule.Import -v
# should trigger a rebuild for a non-module transitive import
test_out "Built Test.NonModule.ImportModuleImport" build Test.NonModule.ImportModuleImport -v

# Cleanup
rm -f produced*
