/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "browser/actions.h"
#include "browser/view.h"
#include "browser/listview.h"
#include "browser/propsplugin.h"
#include "browser/filterwidget.h"

#include "widgets/quickconnect.h"
#include "kftpbookmarks.h"
#include "kftpqueue.h"
#include "misc.h"
#include "kftpapi.h"
#include "kftpsession.h"
#include "misc/config.h"
#include "verifier.h"
#include "mainactions.h"

#include <kglobal.h>
#include <kcharsets.h>
#include <kapplication.h>
#include <kmainwindow.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kinputdialog.h>
#include <kpropertiesdialog.h>
#include <kio/job.h>
#include <kshred.h>
#include <klineedit.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>

#include <qdir.h>
#include <qclipboard.h>

using namespace KFTPGrabberBase;
using namespace KFTPEngine;

namespace KFTPWidgets {

namespace Browser {

Actions::Actions(QObject *parent, const char *name)
 : QObject(parent, name)
{
  m_dirView = static_cast<View*>(parent);
}

KActionCollection *Actions::actionCollection()
{
  return KFTPAPI::getInstance()->mainWindow()->actionCollection();
}

void Actions::initActions()
{
  // Create all the actions
  m_goUpAction = KStdAction::up(this, SLOT(slotGoUp()), actionCollection(), "");
  m_goBackAction = KStdAction::back(this, SLOT(slotGoBack()), actionCollection(), "");
  m_goForwardAction = KStdAction::forward(this, SLOT(slotGoForward()), actionCollection(), "");
  m_goHomeAction = KStdAction::home(this, SLOT(slotGoHome()), actionCollection(), "");

  m_reloadAction = KStdAction::redisplay(this, SLOT(slotReload()), actionCollection(), "");
  m_reloadAction->setText(i18n("&Reload"));
  
  m_abortAction = new KAction(i18n("&Abort"), "stop", KShortcut(), this, SLOT(slotAbort()), actionCollection());
  m_toggleTreeViewAction = new KToggleAction(i18n("&Show Tree View"), "view_tree", KShortcut(), this, SLOT(slotShowHideTree()), actionCollection());
  m_toggleFilterAction = new KToggleAction(i18n("Show &Filter"), "filter", KShortcut(), this, SLOT(slotShowHideFilter()), actionCollection(), "filter");
  
  m_transferAction = new KAction(i18n("&Transfer"), KShortcut(), this, SLOT(slotTransfer()), actionCollection());
  m_queueTransferAction = new KAction(i18n("&Queue Transfer"), "queue", KShortcut(), this, SLOT(slotQueueTransfer()), actionCollection());
  m_createDirAction = new KAction(i18n("&Create Directory..."), "folder_new", KShortcut(), this, SLOT(slotCreateDir()), actionCollection());
  m_fileEditAction = new KAction(i18n("&Open file"), "fileopen", KShortcut(), this, SLOT(slotFileEdit()), actionCollection());
  m_verifyAction = new KAction(i18n("&Verify..."), "ok", KShortcut(), this, SLOT(slotVerify()), actionCollection());

  populateEncodings();
  
  m_moreActions = new KActionMenu(i18n("&More Actions"), "configure", this);
  m_rawCmdAction = new KAction(i18n("Send &Raw Command..."), "openterm", KShortcut(), this, SLOT(slotRawCmd()), actionCollection());
  m_exportListingAction = new KAction(i18n("&Export Directory Listing..."), "", KShortcut(), this, SLOT(slotExportListing()), actionCollection());
  m_showHiddenFilesAction = new KToggleAction(i18n("Show &Hidden Files && Directories"), KShortcut(), this, SLOT(slotShowHiddenFiles()), actionCollection());
  
  m_moreActions->insert(m_rawCmdAction);
  m_moreActions->insert(m_changeEncodingAction);
  m_moreActions->insert(m_exportListingAction);
  m_moreActions->popupMenu()->insertSeparator();
  m_moreActions->insert(m_showHiddenFilesAction);
  
  m_moreActions->setStickyMenu(true);
  m_moreActions->setDelayed(false);

  m_siteChangeAction = new KActionMenu(i18n("&Change Site"), "goto", this);
  m_quickConnectAction = new KAction(i18n("&Quick Connect..."), "connect_creating", KShortcut(), this, SLOT(slotQuickConnect()), actionCollection());
  m_connectAction = new KActionMenu(i18n("&Connect To"), this);
  m_disconnectAction = new KAction(i18n("&Disconnect"), "connect_no", KShortcut(), this, SLOT(slotDisconnect()), actionCollection());

  m_siteChangeAction->insert(m_quickConnectAction);
  m_siteChangeAction->insert(m_connectAction);
  m_siteChangeAction->insert(m_disconnectAction);
  m_siteChangeAction->setStickyMenu(true);
  m_siteChangeAction->setDelayed(false);

  // Populate bookmarks
  KFTPBookmarks::Manager::self()->guiPopulateBookmarksMenu(m_connectAction, QDomNode(), false, m_dirView->m_session);
}

void Actions::populateEncodings()
{
  // Charsets
  m_changeEncodingAction = new KActionMenu(i18n("Change Remote &Encoding"), "charset", actionCollection(), "changeremoteencoding");
  m_changeEncodingAction->setDelayed(false);
  
  KPopupMenu *menu = m_changeEncodingAction->popupMenu();
  menu->clear();
  
  QStringList charsets = KGlobal::charsets()->descriptiveEncodingNames();
  int count = 0;
  for (QStringList::iterator i = charsets.begin(); i != charsets.end(); ++i)
    menu->insertItem(*i, this, SLOT(slotCharsetChanged(int)), 0, ++count);
  
  menu->insertSeparator();
  menu->insertItem(i18n("Default"), this, SLOT(slotCharsetReset(int)), 0, ++count);
  menu->setItemChecked(count, true);
  
  m_defaultCharsetOption = count;
  m_curCharsetOption = count;
}

void Actions::updateActions()
{
  // Enable/disable actions
  m_goUpAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Up));
  m_goBackAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Back));
  m_goForwardAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Forward));
  m_goHomeAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Home));
  m_reloadAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Reload));
  
  m_abortAction->setEnabled(m_dirView->m_ftpClient->socket()->isBusy());
  m_toggleTreeViewAction->setEnabled(true);
  m_toggleFilterAction->setEnabled(true);

  m_quickConnectAction->setEnabled(m_dirView->m_list->m_curURL.isLocalFile());
  m_connectAction->setEnabled(true);
  m_disconnectAction->setEnabled(m_dirView->m_ftpClient->socket()->isConnected());
  
  if (m_dirView->m_list->getCurrentSelection().count() == 1) {
    m_fileEditAction->setEnabled(!m_dirView->m_list->getCurrentSelectionLI().at(0)->m_dirEntry.isDirectory());
    m_verifyAction->setEnabled(m_dirView->m_list->m_curURL.isLocalFile() && m_dirView->m_list->getCurrentSelectionLI().at(0)->m_url.fileName().lower().right(3) == "sfv");
  } else {
    m_fileEditAction->setEnabled(false);
    m_verifyAction->setEnabled(false);
  }

  // We can queue a transfer if we canTransfer()
  if (m_dirView->m_list->canTransfer()) {
    m_queueTransferAction->setEnabled(true);
  } else {
    m_queueTransferAction->setEnabled(false);
  }

  m_transferAction->setEnabled(m_dirView->m_list->canTransfer());
  m_createDirAction->setEnabled(m_dirView->m_list->canLocalActions());

  m_changeEncodingAction->setEnabled(!m_dirView->m_list->m_curURL.isLocalFile());
  m_exportListingAction->setEnabled(true);
  m_rawCmdAction->setEnabled(!m_dirView->m_list->m_curURL.isLocalFile() && 
                             m_dirView->m_ftpClient->socket()->features() & SF_RAW_COMMAND);
  
  // Update the main actions as well
  KFTPAPI::getInstance()->mainWindow()->getActions()->updateActions();
}

void Actions::slotGoUp()
{
  m_dirView->m_list->moveTo(ListView::Up);
}

void Actions::slotGoBack()
{
  m_dirView->m_list->moveTo(ListView::Back);
}

void Actions::slotGoForward()
{
  m_dirView->m_list->moveTo(ListView::Forward);
}

void Actions::slotReload()
{
  m_dirView->m_list->moveTo(ListView::Reload);
}

void Actions::slotGoHome()
{
  m_dirView->m_list->moveTo(ListView::Home);
}

void Actions::slotQuickConnect()
{
  // Create/get the new dialog
  QuickConnectDialog *quickConnect = new QuickConnectDialog(m_dirView);

  if (quickConnect->exec() == KDialogBase::Accepted) {
    // Get the url and connect
    if (m_dirView->m_ftpClient->socket()->isConnected()) {
      if (KFTPCore::Config::confirmDisconnects() && KMessageBox::warningYesNo(0, i18n("Do you want to drop current connection?")) == KMessageBox::No)
        return;
      
      m_dirView->m_session->disconnectAllConnections();
    }

    m_dirView->m_session->setSite(0);
    
    quickConnect->setupClient(m_dirView->m_ftpClient);
    m_dirView->m_ftpClient->connect(quickConnect->getUrl());
  }

  delete quickConnect;
}

void Actions::slotDisconnect()
{
  if (m_dirView->m_ftpClient->socket()->isConnected()) {
    if (KFTPCore::Config::confirmDisconnects() && KMessageBox::warningYesNo(0, i18n("Do you want to drop current connection?")) == KMessageBox::No)
      return;

    m_dirView->m_session->disconnectAllConnections();
  }
}

void Actions::slotTransfer()
{
    // Queue a transfer
  ListItems selection = m_dirView->m_list->getCurrentSelectionLI();
  KFTPQueue::Transfer *transfer = 0L;

  for (unsigned int i = 0; i < selection.count(); i++) {
    ListViewItem *item = selection.at(i);
    KURL destinationUrl = m_dirView->m_list->m_companion->m_curURL;
    destinationUrl.addPath(item->m_dirEntry.filename());

    transfer = KFTPQueue::Manager::self()->spawnTransfer(
      item->m_url,
      destinationUrl,
      item->m_dirEntry.size(),
      item->m_dirEntry.isDirectory(),
      selection.count() == 1,
      true,
      0L,
      true
    );
  }

  // Execute transfer
  if (transfer)
    static_cast<KFTPQueue::Site*>(transfer->parentObject())->delayedExecute();
}

void Actions::slotQueueTransfer()
{
  // Queue a transfer
  ListItems selection = m_dirView->m_list->getCurrentSelectionLI();

  for (unsigned int i = 0; i < selection.count(); i++) {
    ListViewItem *item = selection.at(i);
    KURL destinationUrl = m_dirView->m_list->m_companion->m_curURL;
    destinationUrl.addPath(item->m_dirEntry.filename());

    KFTPQueue::Manager::self()->spawnTransfer(
      item->m_url,
      destinationUrl,
      item->m_dirEntry.size(),
      item->m_dirEntry.isDirectory(),
      selection.count() == 1,
      true,
      0L,
      i != 0
    );
  }
}

void Actions::slotCreateDir()
{
  // Create new directory
  bool ok;
  QString newDirName = KInputDialog::getText(i18n("Create Directory"), i18n("Directory name:"), "", &ok);
  
  if (ok) {
    if (m_dirView->m_list->m_curURL.isLocalFile()) {
      KIO::mkdir(KURL(m_dirView->m_list->m_curURL.path(1) + newDirName));
    } else {
      KURL newDir = m_dirView->m_list->m_curURL;
      newDir.addPath(newDirName);
      m_dirView->m_ftpClient->mkdir(newDir);
    }
  }
}

void Actions::slotFileEdit()
{
  // Get Selection
  ListItems selection = m_dirView->m_list->getCurrentSelectionLI();

  bool isLocal = m_dirView->m_list->m_curURL.isLocalFile();
  ListViewItem *item = selection.at(0);
  
  if (!item->m_dirEntry.isDirectory()) {
    if (isLocal) {
      QString mimeType = KMimeType::findByURL(item->m_url, 0, true, true)->name();
      KService::Ptr offer = KServiceTypeProfile::preferredService(mimeType, "Application");
      
      if (offer)
        KRun::run(offer->exec(), item->m_url, offer->name(), offer->icon());
    } else {
      // Create a new transfer to download the file and open it
      KFTPQueue::TransferFile *transfer = new KFTPQueue::TransferFile(KFTPQueue::Manager::self());
      transfer->setSourceUrl(item->m_url);
      transfer->setDestUrl(KURL(KGlobal::dirs()->saveLocation("tmp") + QString("%1-%2").arg(KApplication::randomString(7)).arg(item->m_url.fileName())));
      transfer->addSize(item->m_dirEntry.size());
      transfer->setTransferType(KFTPQueue::Download);
      transfer->setOpenAfterTransfer(true);
      KFTPQueue::Manager::self()->insertTransfer(transfer);
      
      // Execute the transfer
      transfer->delayedExecute();
    }
  }
}

void Actions::slotAbort()
{
  KFTPSession::Session *session = KFTPSession::Manager::self()->find(m_dirView);

  // Abort the session
  if (session)
    session->abort();
}

void Actions::slotRawCmd()
{
  bool ok;
  QString rawCmd = KInputDialog::getText(i18n("Send Raw Command"), i18n("Command:"), "", &ok);

  if (ok)
    m_dirView->m_ftpClient->raw(rawCmd);
}

void Actions::slotShowHideTree()
{
  m_dirView->setTreeVisible(m_toggleTreeViewAction->isChecked());
}

void Actions::slotShowHideFilter()
{
  if (m_toggleFilterAction->isChecked()) {
    m_dirView->m_searchToolBar->show();
    m_dirView->m_searchFilter->clear();
    m_dirView->m_searchFilter->setFocus();
  } else {
    m_dirView->m_searchFilter->clear();
    m_dirView->m_searchToolBar->hide();
  }
}

void Actions::slotCharsetChanged(int id)
{
  if (!m_changeEncodingAction->popupMenu()->isItemChecked(id)) {
    QStringList charsets = KGlobal::charsets()->descriptiveEncodingNames();
    QString charset = KGlobal::charsets()->encodingForName(charsets[id - 1]);
    
    // Set the current socket's charset
    m_dirView->m_ftpClient->socket()->changeEncoding(charset);
    
    // Update checked items
    m_changeEncodingAction->popupMenu()->setItemChecked(id, true);
    m_changeEncodingAction->popupMenu()->setItemChecked(m_curCharsetOption, false);
    m_curCharsetOption = id;
  }
}

void Actions::slotCharsetReset(int id)
{
  // Revert to default charset if possible
  KFTPBookmarks::Site *site = m_dirView->m_session->getSite();
  
  if (site) {
    // Set the current socket's charset
    m_dirView->m_ftpClient->socket()->changeEncoding(site->getProperty("encoding"));
    
    // Update checked items
    m_changeEncodingAction->popupMenu()->setItemChecked(id, true);
    m_changeEncodingAction->popupMenu()->setItemChecked(m_curCharsetOption, false);
    m_curCharsetOption = id;
  }
}

void Actions::slotExportListing()
{
  QString savePath = KFileDialog::getSaveFileName(QString::null, "*.txt|Directory Dump", 0, i18n("Export Directory Listing"));
  
  if (!savePath.isEmpty()) {
    QFile file(savePath);
    
    if (!file.open(IO_WriteOnly))
      return;
      
    QTextStream stream(&file);
      
    QListViewItem *item = m_dirView->getListView()->firstChild();
    while (item) {
      ListViewItem *i = static_cast<ListViewItem*>(item);
      
      stream << i->m_dirEntry.permissions() << "\t";
      stream << i->m_dirEntry.owner() << "\t" << i->m_dirEntry.group() << "\t";
      stream << i->m_dirEntry.timeAsString() << "\t";
      stream << i->m_dirEntry.filename() << "\t";
      stream << "\n";
      
      item = item->nextSibling();
    }
      
    file.flush();
    file.close();
  }
}

void Actions::slotVerify()
{
  KFTPWidgets::Verifier *verifier = new KFTPWidgets::Verifier();
  verifier->setFile(m_dirView->m_list->getCurrentSelection()[0].path());
  verifier->exec();
  
  delete verifier;
}

void Actions::slotShowHiddenFiles()
{
  m_dirView->m_list->setShowHidden(m_showHiddenFilesAction->isChecked());
  m_dirView->m_list->moveTo(ListView::Reload);
}

}

}

#include "actions.moc"
