/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "browser/listview.h"
#include "browser/treeview.h"
#include "browser/view.h"
#include "browser/actions.h"

#include "misc.h"
#include "misc/config.h"
#include "misc/filter.h"

#include "engine/cache.h"
#include "kftpqueue.h"
#include "kftpsession.h"
#include "kftpbookmarks.h"
#include "kftpapi.h"

#include <klocale.h>
#include <kmessagebox.h>
#include <kiconeffect.h>
#include <kcombobox.h>
#include <kpopupmenu.h>
#include <kio/job.h>

#include <qheader.h>
#include <qdir.h>
#include <qimage.h>
#include <qpainter.h>
#include <qapplication.h>
#include <qregexp.h>

using namespace KFTPGrabberBase;
using namespace KFTPEngine;
using namespace KFTPCore::Filter;

namespace KFTPWidgets {

namespace Browser {

/*
 *                      History class
 */

History::History()
  : m_historyPos(0)
{
}
 
void History::newEntry(const KURL &url)
{
  // Add an entry to the list
  if (m_historyPos != m_history.fromLast() && m_history.count() > 1) {
    while (m_historyPos != m_history.fromLast()) {
      m_history.pop_back();
    }
  }

  if (m_history.empty() || (*m_historyPos).first != url) {
    // Add the currently selected URL if there is some history
    if (!m_history.empty())
      (*m_historyPos).second = url.fileName();

    m_history.append(QPair<KURL, QString>(url, QString::null));
  }

  // Set the new history position
  m_historyPos = m_history.fromLast();
}

KURL History::getCurrent()
{
  return (*m_historyPos).first;
}

QString History::getSelected()
{
  return (*m_historyPos).second;
}

void History::goBack()
{
  if (m_historyPos != m_history.begin())
    m_historyPos--;
}

void History::goForward()
{
  if (m_historyPos != m_history.fromLast())
    m_historyPos++;
}

/*
 *                        ListViewItem class
 */

ListViewItem::ListViewItem(ListView *parent)
  : KListViewItem(parent)
{
}

ListViewItem::ListViewItem(ListViewItem *parent)
  : KListViewItem(parent)
{
}

ListViewItem::ListViewItem(ListView *parent, KFileItem *item)
  : KListViewItem(parent)
{
  readFromFileItem(item);
}

ListViewItem::ListViewItem(ListViewItem *parent, KFileItem *item)
  : KListViewItem(parent)
{
  readFromFileItem(item);
}

ListViewItem::ListViewItem(ListView *parent, const DirectoryEntry &item, const KURL &url)
  : KListViewItem(parent),
    m_dirEntry(item),
    m_url(url)
{
  createColumns();
}

void ListViewItem::readFromFileItem(KFileItem *item)
{
  // Transfer all item's properties to our class
  m_url = item->url();

  m_dirEntry.setPermissions(item->permissionsString());
  m_dirEntry.setOwner(item->user());
  m_dirEntry.setGroup(item->group());
  
  if (item->isDir())
    m_dirEntry.setType('d');
  else
    m_dirEntry.setType('f');
    
  m_dirEntry.setLink(item->linkDest());
  m_dirEntry.setFilename(item->name());
  m_dirEntry.setTime(item->time(KIO::UDS_MODIFICATION_TIME));
  m_dirEntry.setSize(item->size());

  // FileItem ptr (just for later identification)
  m_fileItem = item;

  // Create the columns
  createColumns();
}

void ListViewItem::createColumns()
{
  // Create the columns from m_dirObject
  setText(0, m_dirEntry.filename());
  setText(2, m_dirEntry.timeAsString());
  setText(3, m_dirEntry.permissions());
  setText(4, m_dirEntry.owner());
  setText(5, m_dirEntry.group());

  QString iconText;
  if (!m_url.isLocalFile()) {
    // Guess file type
    if (m_dirEntry.isDirectory()) {
      iconText = "folder";
    } else if (m_dirEntry.isSymlink()) {
      // We can't know if the sym-linked file is realy a directory, but most of
      // the time it is. So if we can't determine the MIME type, set it to directory.
      KMimeType::Ptr mimeType = KMimeType::findByURL(m_url, 0, false, true);
      
      if (mimeType->name() == KMimeType::defaultMimeType()) {
        iconText = "folder";
        m_dirEntry.setType('d');
      } else {
        iconText = mimeType->icon(QString::null, false);
      }
    } else {
      KMimeType::Ptr mimeType = KMimeType::findByURL(m_url, 0, false, true);
      iconText = mimeType->icon(QString::null, false);
    }
  }
  
  if (m_dirEntry.isDirectory())
    setText(1, " - ");
  else {
    QString sizeText = KFTPCore::Config::showSizeInBytes() ? KIO::number(m_dirEntry.size()) : KIO::convertSize(m_dirEntry.size());
    sizeText.append(" ");
    setText(1, sizeText);
  }

  // Get the pixmap
  QPixmap filePixmap;
  
  if (!m_url.isLocalFile()) {
    KMimeType::Ptr mimeType = KMimeType::findByURL(m_url, 0, m_url.isLocalFile(), !m_url.isLocalFile());
    filePixmap = loadSmallPixmap(iconText);
  
    if (m_dirEntry.isSymlink()) {
      // Do the link icon overlay for symlinks
      QImage linkImage = loadSmallPixmap("link_overlay").convertToImage();
      QImage fileImage = filePixmap.convertToImage();
      KIconEffect::overlay(fileImage, linkImage);
      filePixmap.convertFromImage(fileImage);
    }
  } else {
    // Use the pixmap provided by the KFileItem
    filePixmap = m_fileItem->pixmap(16);
  }

  setPixmap(0, filePixmap);
}

int ListViewItem::compare(QListViewItem *i, int col, bool) const
{
  ListViewItem *cmp = static_cast<ListViewItem*>(i);
  
  if (m_dirEntry.isDirectory() != cmp->m_dirEntry.isDirectory())
    return m_dirEntry.isDirectory() ? -1 : 1;
    
  bool isHidden1 = m_dirEntry.filename().at(0) == '.';
  bool isHidden2 = cmp->m_dirEntry.filename().at(0) == '.';
    
  if (isHidden1 != isHidden2)
    return isHidden1 ? -1 : 1;
  
  switch (col) {
    case 2: {
      // Sorting by time
      time_t t1 = m_dirEntry.time();
      time_t t2 = cmp->m_dirEntry.time();
      
      return (t1 > t2) ? 1 : (t1 < t2) ? -1 : 0;
    }
  }

  return text(col).lower().localeAwareCompare(cmp->text(col).lower());
}

void ListViewItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int alignment)
{
  QColorGroup colorGroup(cg);
  QColor textColor = colorGroup.text();

  // Set custom file item color if set
  if (m_textColor.isValid())
    colorGroup.setColor(QColorGroup::Text, m_textColor);

  QListViewItem::paintCell(p, colorGroup, column, width, alignment);
  
  if (column < listView()->columns() - 1) {
    // Draw a separator between columns
    p->setPen(KGlobalSettings::buttonBackground());
    p->drawLine(width - 1, 0, width - 1, height() - 1);
  }
}

/*
 *                          ListView class
 */

ListView::ListView(QWidget *parent, KFTPSession::Session *session)
 : KFTPWidgets::ListView(parent),
   m_session(session),
   m_ftpClient(session->getClient()),
   m_shouldDisableResize(true),
   m_autoResizeEnabled(true)
{
  // Create the columns
  addColumn(i18n("Name"));
  addColumn(i18n("Size"));
  addColumn(i18n("Date"));
  addColumn(i18n("Permissions"));

  // Set column width
  setColumnWidthMode(0, QListView::Manual);
  setColumnWidthMode(1, QListView::Manual);
  setColumnWidthMode(2, QListView::Manual);
  setColumnWidthMode(3, QListView::Manual);
  
  // Set column alignments
  setColumnAlignment(1, Qt::AlignRight);
  setColumnAlignment(2, Qt::AlignHCenter);
  setColumnAlignment(3, Qt::AlignHCenter);
  
  if (KFTPCore::Config::showOwnerGroup()) {
    // Only show owner/group if set in the config
    addColumn(i18n("Owner"));
    addColumn(i18n("Group"));
    
    setColumnWidthMode(4, QListView::Manual);
    setColumnWidthMode(5, QListView::Manual);
    
    setColumnAlignment(4, Qt::AlignHCenter);
    setColumnAlignment(5, Qt::AlignHCenter);
  } else {
    setFullWidth(true);
  }
  
  setColumnWidth(0, 140);
  setColumnWidth(2, 100);

  // Other settings
  setDragEnabled(true);
  setAcceptDrops(true);
  setAllColumnsShowFocus(true);
  setSelectionModeExt(Extended);
  setEmptyListText("This directory is empty or the contents\n could not be displayed.");
  setHScrollBarMode(QScrollView::AlwaysOff);
  
  QHeader *viewHeader = header();
  viewHeader->setResizeEnabled(true);
  viewHeader->setMovingEnabled(false);
  
  connect(viewHeader, SIGNAL(sizeChange(int, int, int)), this, SLOT(slotHeaderResized(int)));

  // Create the dir lister for local files and connect signals
  m_dirLister = new KDirLister();

  connect(m_dirLister, SIGNAL(newItems(const KFileItemList&)), this, SLOT(slotDirListerNewItems(const KFileItemList&)));
  connect(m_dirLister, SIGNAL(deleteItem(KFileItem*)), this, SLOT(slotDirListerDeleteItem(KFileItem*)));
  connect(m_dirLister, SIGNAL(clear()), this, SLOT(slotDirListerClear()));
  connect(m_dirLister, SIGNAL(refreshItems(const KFileItemList&)), this, SLOT(slotDirListerRefreshItems(const KFileItemList&)));

  // Connect the ftp-related signals
  connect(m_ftpClient->eventHandler(), SIGNAL(engineEvent(KFTPEngine::Event*)), this, SLOT(slotFtpEngineEvent( KFTPEngine::Event*)));

  // Connect the list-view related signals
  connect(this, SIGNAL(executed(QListViewItem*)), this, SLOT(slotListClicked()));
  connect(this, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)), this, SLOT(slotContextMenu(QListViewItem*, const QPoint&)));
  connect(this, SIGNAL(itemRenamed(QListViewItem*, const QString&, int)), this, SLOT(slotItemRenamed(QListViewItem*, const QString&, int)));

  // Create the timer for drag & drop
  m_dropItem = 0L;
  m_dragOpenTimer = new QTimer(this);
  connect(m_dragOpenTimer, SIGNAL(timeout()), this, SLOT(slotDragTimeout()));

  // Set the default home url
  setHomeUrl(KURL(KFTPCore::Config::defLocalDir()));
  setShowHidden(KFTPCore::Config::showHiddenFiles());

  // Reset companion
  setCompanion(0);
}

ListView::~ListView()
{
  if (m_dirLister)
    delete m_dirLister;
}

void ListView::setTreeView(TreeView *tree)
{
  m_tree = tree;

  // Connect the signals
  connect(m_tree, SIGNAL(pathChanged(const KURL)), this, SLOT(slotTreeClicked(const KURL)));
}

void ListView::resetView(const KURL &url, const QString &rootText)
{
  // Clear all the views
  m_tree->resetView(url, rootText);
  clear();

  // Clear history
  m_history.m_history.clear();
  m_history.m_historyPos = m_history.m_history.fromLast();
  static_cast<View*>(parent()->parent())->m_historyCombo->clear();
}

bool ListView::canTransfer(ListView *caller)
{
  // First check that we are transfer capable
  if (!canLocalActions())
    return false;

  // If there is no companion there shall be no transfers
  if (!m_companion && !caller)
    return false;

  // If both (this and the companion) are local, no transfer is possible
  if (m_companion->m_curURL.isLocalFile() && m_curURL.isLocalFile())
    return false;
    
  // If both are remote and protocols differ or one protocol doesn't support site-to-site
  // transfers, no transfer is possible
  if (!m_companion->m_curURL.isLocalFile() && !m_curURL.isLocalFile() &&
      (
        m_companion->m_ftpClient->socket()->protocolName() != m_ftpClient->socket()->protocolName() ||
        !(m_companion->m_ftpClient->socket()->features() & SF_FXP_TRANSFER) ||
        !(m_ftpClient->socket()->features() & SF_FXP_TRANSFER)
      ))
    return false;

  // Check if the other end is ready
  if (caller) {
    // This has already been called by our companion
    if (m_curURL.isLocalFile())
      return true;
    else if (m_ftpClient->socket()->isConnected())
      return true;
  } else {
    // We check the companion first
    if (m_companion->canTransfer(this)) {
      // We are transfer capable
      return true;
    }
  }

  return false;
}

bool ListView::canLocalActions()
{
  if (m_curURL.isLocalFile())
    return true;
  else if (m_ftpClient->socket()->isConnected())
    return true;

  return false;
}

int ListView::transferMode()
{
  // Check what kind of url does our companion have
  bool c_local = m_companion->m_curURL.isLocalFile();
  bool w_local = m_curURL.isLocalFile();

  // Now analyze :)
  if (c_local && w_local) /* Two local lists ?!? WIERD */
    return -1;

  if (c_local && !w_local || w_local && !c_local) /* One local and one non-local */
    return 1;

  if (!c_local && !w_local) /* Two remotes - fxp transfer mode */
    return 2;

  // This should not happen
  return -1;
}

KURL::List ListView::getCurrentSelection()
{
  // Get all the files that are currently selected
  KURL::List list;

  // Go trough all items and add them to the list
  for (unsigned int i = 0; i < KListView::selectedItems().count(); i++) {
    ListViewItem *item = (ListViewItem*) KListView::selectedItems().at(i);
    list.append(item->m_url);
  }

  return list;
}

ListItems ListView::getCurrentSelectionLI()
{
  // Get all the files that are currently selected
  ListItems list;

  // Go trough all items and add them to the list
  for (unsigned int i = 0; i < KListView::selectedItems().count(); i++) {
    ListViewItem *item = (ListViewItem*) KListView::selectedItems().at(i);
    list.append(item);
  }

  return list;
}

void ListView::slotFtpEngineEvent(KFTPEngine::Event *event)
{
  switch (event->type()) {
    case Event::EventConnect: {
      // ***************************************************************************
      // ****************************** EventConnect *******************************
      // ***************************************************************************
      if (m_session->getSite() && !m_session->getSite()->getProperty("defremotepath").isEmpty())
        m_homeURL = m_session->getSite()->getProperty("defremotepath");
      else
        m_homeURL = m_ftpClient->socket()->getDefaultDirectory();
  
      KURL newURL = m_ftpClient->socket()->getCurrentUrl();
      newURL.setPath(m_homeURL.path());
      m_homeURL = newURL;
      m_curURL = newURL;
      
      resetView(newURL, newURL.host());
      openUrl(newURL);
      
      if (m_companion && m_companion->m_curURL.isLocalFile() && m_session->getSite()) {
        QString localPath = m_session->getSite()->getProperty("deflocalpath");
        
        if (!localPath.isEmpty())
          m_companion->openUrl(localPath);
      }
      break;
    }
    case Event::EventDisconnect: {
      // ***************************************************************************
      // **************************** EventDisconnect ******************************
      // ***************************************************************************
      
      // Change to local root
      openUrl(KURL(KFTPCore::Config::defLocalDir()));
      break;
    }
    case Event::EventError: {
      // ***************************************************************************
      // ******************************* EventError ********************************
      // ***************************************************************************
      ErrorCode error = event->getParameter(0).asErrorCode();
      
      if (error == ListFailed) {
        // We are unable to enter a directory, display an error message
        KComboBox *historyCombo = static_cast<View*>(parent()->parent())->m_historyCombo;
        historyCombo->setCurrentItem(m_curURL.path());
        
        // Remove this url from KHistoryCombo
        QString errorDirectory = m_ftpClient->socket()->getCurrentDirectory();
        for (int i = 0; i < historyCombo->count(); i++) {
          if (historyCombo->text(i) == errorDirectory)
            historyCombo->removeItem(i);
        }
        
        KMessageBox::error(0, i18n("Unable to open directory '%1'.").arg(errorDirectory));
      }
      break;
    }
    case Event::EventDirectoryListing: {
      // ***************************************************************************
      // ************************** EventDirectoryListing **************************
      // ***************************************************************************
      KURL url = m_reqURL;

      // Disable dirlister for the time of remote connections ;)
      m_dirLister->setAutoUpdate(false);

      KListView::clear();
      QValueList<DirectoryEntry> list = event->getParameter(0).asDirectoryListing().list();

      // Check the file count
      if (list.count() > 2000) {
        // If there are more than 2000 files in the dir, ask the user if the files
        // should be displayed
        QString warnText;
        warnText  = i18n("This directory contains 1 file.", "This directory contains %n files.", list.count());
        warnText += " ";
        warnText += i18n("Displaying all might take some time.");
        warnText += "\n\n";
        warnText += i18n("Are you sure?");

        if (KMessageBox::warningContinueCancel(0, warnText, QString::null, KStdGuiItem::cont(), "ManyFilesWarning") == KMessageBox::Cancel)
        {
          // Nope, don't display them ;)
          if (canMoveTo(Back)) {
            moveTo(Back);
          } else if (url.path() != "/") {
            KURL newURL = m_curURL;
            newURL.setPath("/");
            openUrl(newURL);
          } else {
            // Nothing left to do :(
          }
          return;
        }
      }
      
      // Create the folder if it doesn't exist yet
      m_tree->createFolder(url, loadSmallPixmap("folder"));

      // Generate file url
      KURL fileUrlBase = m_ftpClient->socket()->getCurrentUrl();

      // Process the directory listing
      int curNum = 0;
      QValueList<DirectoryEntry>::iterator end(list.end());
      for (QValueList<DirectoryEntry>::iterator i(list.begin()); i != end; ++i) {
        fileUrlBase.setPath(url.path(1) + (*i).filename());
        ListViewItem *item = new ListViewItem(this, (*i), fileUrlBase);
        
        const ActionChain *actionChain = Filters::self()->process((*i));
        const Action *action;
        
        if (actionChain && (action = actionChain->getAction(Action::Colorize))) {
          item->setColor(action->value().toColor());
          item->repaint();
        }
        
        if ((!m_showHiddenFiles && (*i).filename().at(0) == '.') || (actionChain && actionChain->getAction(Action::Hide))) {
          delete item;
          continue;
        }
        
        // If we have a tree view and this is a directory
        if (item->m_dirEntry.isDirectory() && m_tree)
          m_tree->createFolder(fileUrlBase, *item->pixmap(0));

        // Process events every 500 files (show progress)
        if (++curNum >= 500) {
          curNum = 0;
          qApp->processEvents();
        }
      }
      
      m_tree->endUpdate(url);

      // Set the correct url
      fileUrlBase.setPath(url.path(0));

      // Update the history
      if (m_addHistory) {
        m_history.newEntry(fileUrlBase);

        // Get the KHistoryCombo
        KComboBox *histCombo = static_cast<View*>(parent()->parent())->m_historyCombo;

        // Remove duplicates from KHistoryCombo
        for(int i = 0;i<histCombo->count();i++) {
          if (histCombo->text(i) == fileUrlBase.path(-1))
            histCombo->removeItem(i);
        }

        // Add to KHistoryCombo
        histCombo->insertItem(loadSmallPixmap("folder"), fileUrlBase.path(-1), 0);
        histCombo->setCurrentItem(0);
      }

      // Set the current url
      m_curURL = fileUrlBase;
      
      // Reselect the correct folder
      smartFolderReselect();

      // Request update
      static_cast<View*>(parent()->parent())->updateActions();

      // Open the tree
      m_tree->openUrl(fileUrlBase);
      
      emit itemsChanged();
      break;
    }
    default: break;
  }
}

void ListView::openUrl(const KURL &url, bool addHistory)
{
  // Detect site change
  KURL tmp = m_curURL;
  tmp.setPath(url.path());
  
  if (tmp != url) {
    // Site change, resetting views
    if (url.isLocalFile()) {
      tmp = KURL(KFTPCore::Config::defLocalDir());
      setHomeUrl(KURL(KFTPCore::Config::defLocalDir()));
    } else {
      tmp = m_ftpClient->socket()->getCurrentUrl();
      tmp.setPath(m_homeURL.path());
    }

    resetView(tmp, url.isLocalFile() ? i18n("Root directory") : url.host());
  }

  if (url.isLocalFile()) {
    // We must set this url or the dir lister won't list anything
    m_curURL = url;
    
    // Update the history
    if (addHistory)
      m_history.newEntry(url);
    
    KComboBox *histCombo = static_cast<View*>(parent()->parent())->m_historyCombo;
  
    // Remove duplicates from KHistoryCombo
    for(int i = 0;i<histCombo->count();i++) {
      if (histCombo->text(i) == url.path(-1))
        histCombo->removeItem(i);
    }
  
    // Add to KHistoryCombo
    histCombo->insertItem(loadSmallPixmap("folder"), url.path(-1), 0);
    histCombo->setCurrentItem(0);
    
    // Open the tree
    m_tree->openUrl(url);
    
    m_dirLister->setAutoUpdate(true);
    m_dirLister->setShowingDotFiles(m_showHiddenFiles);
    m_dirLister->emitChanges();
    m_dirLister->openURL(url, false, true);
    
    // Reselect the correct folder
    smartFolderReselect();
    
    // Request update
    static_cast<View*>(parent()->parent())->updateActions();
  } else if (m_ftpClient) {
    // Request the dir listing and return
    m_reqURL = url;
    m_addHistory = addHistory;
    
    m_ftpClient->list(url);
  }
}

void ListView::smartFolderReselect()
{
  // Scroll to the item that was previously selected
  if (m_history.getSelected() != QString::null) {
    QListViewItem *i = findItem(m_history.getSelected(), 0);
    
    if (i) {
      setSelected(i, true);
      ensureItemVisible(i);
    }
  } else if (canMoveTo(Back)) {
    // Did we just go up ?
    m_history.goBack();
    
    if (m_curURL == m_history.getCurrent().upURL()) {
      QListViewItem *i = findItem(m_history.getCurrent().fileName(), 0);
    
      if (i) {
        setSelected(i, true);
        ensureItemVisible(i);
      }
    }
    
    m_history.goForward();
  }
}

void ListView::moveTo(LocationInfo location)
{
  switch (location) {
    case Up: {
      // Just get the one directory up and reload
      openUrl(m_curURL.upURL());
      break;
    }
    case Back: {
      // Use the history and go back
      m_history.goBack();
      openUrl(m_history.getCurrent(), false);
      break;
    }
    case Forward: {
      // Use the history and go forward
      m_history.goForward();
      openUrl(m_history.getCurrent(), false);
      break;
    }
    case Reload: {
      // Clear the cache and reload the current url
      Cache::self()->invalidateEntry(m_curURL);
      openUrl(m_curURL, false);
      break;
    }
    case Home: {
      // Go home =)
      openUrl(m_homeURL);
      break;
    }
  }
}

bool ListView::canMoveTo(LocationInfo location)
{
  switch (location) {
    case Up: {
      if (m_curURL.path() == "/")
        return false;
      break;
    }
    case Back: {
      if (m_history.m_historyPos == m_history.m_history.begin())
        return false;
      break;
    }
    case Forward: {
      if (m_history.m_historyPos == m_history.m_history.fromLast())
        return false;
      break;
    }
    case Home:
    case Reload: {
      if (!m_curURL.isLocalFile() && !m_ftpClient->socket()->isConnected())
        return false;
      break;
    }
  }

  return true;
}

void ListView::slotContextMenu(QListViewItem *i, const QPoint &p)
{
  // Request update
  static_cast<View*>(parent()->parent())->updateActions();

  // Create the popup menu
  KPopupMenu *contextMenu = new KPopupMenu(this);
  
  KActionCollection *mActions = KFTPAPI::getInstance()->mainWindow()->actionCollection();
  Actions *actions = static_cast<View*>(parent()->parent())->m_actions;

  // Always show create directory
  actions->m_createDirAction->plug(contextMenu);
  contextMenu->insertSeparator();

  // If nothing is selected, show the navigation menus
  if (!i) {
    actions->m_goUpAction->plug(contextMenu);
    actions->m_goBackAction->plug(contextMenu);
    actions->m_goForwardAction->plug(contextMenu);
    actions->m_reloadAction->plug(contextMenu);
  } else {
    actions->m_transferAction->plug(contextMenu);
    actions->m_queueTransferAction->plug(contextMenu);
    mActions->action("edit_rename")->plug(contextMenu);
    mActions->action("edit_delete")->plug(contextMenu);
    mActions->action("edit_shred")->plug(contextMenu);
    actions->m_fileEditAction->plug(contextMenu);
    actions->m_verifyAction->plug(contextMenu);
    contextMenu->insertSeparator();
    mActions->action("edit_copy")->plug(contextMenu);
    mActions->action("edit_paste")->plug(contextMenu);
    contextMenu->insertSeparator();
    mActions->action("edit_filter_options")->plug(contextMenu);
  }

  // Always show properties
  contextMenu->insertSeparator();
  mActions->action("edit_properties")->plug(contextMenu);

  contextMenu->exec(p);
}

void ListView::slotItemRenamed(QListViewItem *item, const QString &str, int)
{
  // Item was renamed, commit the change to the actual filesystem
  ListViewItem *o_item = static_cast<ListViewItem*>(item);
  
  // Get the new file URL
  KURL oldName = o_item->m_url.url();
  KURL newName = oldName;
  newName.setPath(oldName.directory(false) + str);
  
  // Do the actual renaming
  if (oldName.isLocalFile()) {
    KIO::rename(oldName, newName, false);
    
    // Reload the listing
    moveTo(Reload);
  } else if (oldName != newName) {
    m_ftpClient->rename(oldName, newName);
  }
}

void ListView::slotListClicked()
{
  // First, get all selected files
  for (unsigned int i = 0; i < KListView::selectedItems().count(); i++) {
    ListViewItem *item = static_cast<ListViewItem*>(KListView::selectedItems().at(i));

    // Check if this is a directory
    if (item->m_dirEntry.isDirectory()) {
      openUrl(item->m_url);
      return;
    }
  }
}

void ListView::slotTreeClicked(const KURL url)
{
  // Tree was clicked and url was changed, refresh
  openUrl(url);
}

void ListView::slotDirListerNewItems(const KFileItemList &items)
{
  if (!m_curURL.isLocalFile())
    return;

  // Add local files receieved from the dir lister
  KFileItemList p_items = items; // FIXME any better way ?
  for (KFileItem *item = p_items.first(); item; item = p_items.next()) {
    ListViewItem *i = new ListViewItem(this, item);
    
    const ActionChain *actionChain = Filters::self()->process(i->getDirectoryEntry());
    const Action *action;
    
    if (actionChain && (action = actionChain->getAction(Action::Colorize))) {
      i->setColor(action->value().toColor());
      i->repaint();
    }

    if (actionChain && actionChain->getAction(Action::Hide)) {
      delete i;
    } else {
      // If we have a tree view and this is a directory
      if (item->isDir() && m_tree)
        m_tree->createFolder(item->url(), item->pixmap(16));
    }
  }
  
  emit itemsChanged();
}

void ListView::slotDirListerDeleteItem(KFileItem *item)
{
  if (!m_curURL.isLocalFile())
    return;

  // Find the item and delete it - search by URL
  QListViewItem *i = firstChild();
  while (i) {
    if (static_cast<ListViewItem*>(i)->m_fileItem == item) {
      // Remove from tree view as well
      if (m_tree && item->isDir())
        m_tree->removeFolder(item->url());

      delete i;
      break;
    }

    i = i->itemBelow();
  }
}

void ListView::slotDirListerClear()
{
  if (!m_curURL.isLocalFile())
    return;

  // Clear all items
  clear();
}

void ListView::slotDirListerRefreshItems(const KFileItemList &items)
{
  if (!m_curURL.isLocalFile())
    return;
    
  // Refresh the items by deleting and recreating
  KFileItemList p_items = items; // FIXME any better way ?
  for (KFileItem *item = p_items.first(); item; item = p_items.next()) {
    slotDirListerDeleteItem(item);
    ListViewItem *i = new ListViewItem(this, item);
    
    const ActionChain *actionChain = Filters::self()->process(i->getDirectoryEntry());
    const Action *action;
    
    if (actionChain && (action = actionChain->getAction(Action::Colorize))) {
      i->setColor(action->value().toColor());
      i->repaint();
    }

    if (actionChain && actionChain->getAction(Action::Hide)) {
      delete i;
    } else {
      // If we have a tree view and this is a directory
      if (item->isDir() && m_tree)
        m_tree->createFolder(item->url(), item->pixmap(16));
    }
  }
  
  emit itemsChanged();
}

void ListView::slotHeaderResized(int section)
{
  if (m_shouldDisableResize && section == 0) {
    setHScrollBarMode(QScrollView::Auto);
    m_autoResizeEnabled = false;
  }
}

void ListView::resizeContents(int width, int height)
{
  m_shouldDisableResize = false;
  
  KFTPWidgets::ListView::resizeContents(width, height);
  
  // Update the column widths
  if (m_autoResizeEnabled)
    updateColumnWidths();
  
  m_shouldDisableResize = true;
}

void ListView::updateColumnWidths()
{
  // The code below is based on Dolphin, Copyright (C) 2006 by Peter Penz
  const int columnCount = columns();
  int requiredWidth = 0;
  
  for (int i = 1; i < columnCount; ++i) {
    // When a directory contains no items, a minimum width for
    // the column must be available, so that the header is readable.
    int columnWidth = 64;
    QFontMetrics fontMetrics(font());
    
    for (QListViewItem* item = firstChild(); item != 0; item = item->nextSibling()) {
      const int width = item->width(fontMetrics, this, i);
      
      if (width > columnWidth) {
        columnWidth = width;
      }
    }
    
    // Add custom margin
    columnWidth += 16;
    setColumnWidth(i, columnWidth);
    requiredWidth += columnWidth;
  }

  // Resize the first column in a way that the whole available width is used
  int firstColumnWidth = visibleWidth() - requiredWidth;
  if (firstColumnWidth < 128) {
    firstColumnWidth = 128;
  }
  
  setColumnWidth(0, firstColumnWidth);
}

/* Drag & drop support below */
void ListView::startDrag()
{
  dragObject()->drag();
}

bool ListView::acceptDrag(QDropEvent *e)
{
  return KURLDrag::canDecode(e) &&
         ( e->action() == QDropEvent::Copy
           || e->action() == QDropEvent::Move
           || e->action() == QDropEvent::Link )
         && acceptDrops()
         && dragEnabled()
         && e->source() != this;
}

QDragObject *ListView::dragObject()
{
  KURL::List p_selection = getCurrentSelection();
  ListItems p_li = getCurrentSelectionLI();

  // Set the correct pixmap
  QPixmap pix;
  if (p_selection.count() > 1) {
    // More than one object selected
    pix = DesktopIcon("kmultiple", 16);
  } else {
    // One object selected, get it's icon from the only item that is
    // selected
    pix = *p_li.at(0)->pixmap(0);
  }

  // Add some metadata for all urls
  KIO::MetaData p_meta;
  ListViewItem *i;
  for (i = p_li.first(); i; i = p_li.next()) {
    char type = i->m_dirEntry.isDirectory() ? 'D' : 'F';
    QString data;
    data.sprintf("%c:%s", type, KIO::number(i->m_dirEntry.size()).ascii());
    p_meta.insert(i->m_url.htmlURL().local8Bit(), data);
  }

  m_dragObject = new KURLDrag(p_selection, p_meta, this, name());
  m_dragObject->setPixmap(pix, QPoint(pix.width() / 2, pix.height() / 2));
  
  return m_dragObject;
}

void ListView::slotDragTimeout()
{
  m_dragOpenTimer->stop();
  if (!m_dropItem)
    return;

  QListViewItem *i = firstChild();
  while (i) {
    ListViewItem *item = static_cast<ListViewItem*>(i);
    if (i->text(0) == m_dropItem->text(0)) {
      if (item->m_dirEntry.isDirectory())
        openUrl(item->m_url);

      return;
    }

    i = i->itemBelow();
  }
}

void ListView::contentsDragEnterEvent(QDragEnterEvent *e)
{
  if (!acceptDrag(e)) {
    e->accept(false);
    return;
  }

  e->acceptAction();
  QListViewItem *i = itemAt(contentsToViewport(e->pos()));
  if (i) {
    m_dropItem = i;
    m_dragOpenTimer->start(1500);
  }
}

void ListView::contentsDragMoveEvent(QDragMoveEvent *e)
{
  if (!acceptDrag(e)) {
    e->accept(false);
    return;
  }

  e->acceptAction();
  QListViewItem *i = itemAt(contentsToViewport(e->pos()));
  if (i) {
    if (i != m_dropItem) {
      m_dragOpenTimer->stop();
      m_dropItem = i;
      m_dragOpenTimer->start(1500);
    }
  } else {
    m_dragOpenTimer->stop();
  }
}

void ListView::contentsDragLeaveEvent(QDragLeaveEvent*)
{
  m_dragOpenTimer->stop();
  m_dropItem = 0L;
}

void ListView::contentsDropEvent(QDropEvent *e)
{
  m_dragOpenTimer->stop();
  m_dropItem = 0L;

  if (!acceptDrag(e)) {
    e->acceptAction(false);
    return;
  }
  e->acceptAction();

  // Decode the data and try to init transfer
  KIO::MetaData p_meta;
  KURL::List p_urls;
  KURLDrag::decode(e, p_urls, p_meta);

  // Add destination url and call the QueueManager
  p_meta.insert("DestURL", m_curURL.url());
  KURLDrag *drag = new KURLDrag(p_urls, p_meta, this, name());
  KFTPQueue::Manager::self()->insertTransfer(drag);
}

}

}

#include "listview.moc"
