/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2002 by Marco Wegner <mail@marcowegner.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */


#ifndef KBMAILER_H
#define KBMAILER_H

class QString;
class QStringList;

class KConfig;


class KBabelMailer
{
  public:
    KBabelMailer();
    ~KBabelMailer();

    /**
     * Send only one file as a mail attachment. The file can either be sent
     * as a compressed (.tar.bz2) or an uncompressed (.po) file.
     *
     * @param fileName the name of the file to be sent.
     */
    void sendOneFile(const QString& fileName);

    /**
     * Send a list of files as a mail attachment. The files will be
     * included in an archive (.tar.bz2).
     *
     * @param fileList contains the names of the files to be sent.
     * @param initialName the possible name of the archive
     */
    void sendFiles(QStringList fileList, const QString& initialName = QString::null);

    /**
     * The archives which are to be sent as mail attachments are created as
     * temporary files. The names of the successfully created archives are
     * stored. When leaving KBabel these files have to be deleted. This is
     * done here.
     */
    void removeTempFiles();


  private:
    /**
     * This is where the archive is actually created.
     *
     * @param fileList contains the names of the files to be included in
     *  the archive.
     * @param initialName the name used as initial name when the archive
     *  name is queried.
     *
     * @return name of the archive if it was created successfully,
     *  otherwise QString::null.
     */
    QString createArchive(QStringList fileList, QString initialName);

    /**
     * Read the config file to extract the values for compression
     * method and compression of single files.
     */
    void readConfig();

    /**
     * This variable is used to store the pathe to the temp dir.
     */
    QString tempDir;

    /** 
     * This variable is used to store the filenames of the temporary
     * archive files.
     */
    QStringList tempFileNames;

    /**
     * Whether to use bzip2 as compression method. If false use gzip.
     */
    bool bzipCompression;

    /**
     * Whether to compress single files as well.
     */
    bool singleFileCompression;

    /**
     * KConfig object.
     */
    KConfig* config;
};

#endif // KBMAILER_H
