#!/usr/bin/perl

#   Copyright (C) 2011 Mauro Carvalho Chehab
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, version 2 of the License.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
# This small script parses register dumps generated by em28xx driver
# with debug options enabled, generating a source code with the results
# of the dump.
#
# It is a variant of parse_az6007.pl, meant to be used to decode az6007
# register read/writes, with DRX-K
#

use strict;
use Switch;
use Getopt::Long;

my $show_mt2063 = 0;
my $show_drxk = 0;
my $show_other_reqs = 0;
my $show_other_xfer = 0;
my $show_other_lines = 0;	# Useful on some cases
my $show_timestamp = 0;
my $show_all = 0;
my $force_wrt_blocks = 0;

my $argerr = "Invalid arguments.\nUse $0 [--show_all] [--show_drxk] [--show_mt2063] [--show_other_reqs] [--show_other_xfer] [--show_other_lines] [--show_timestamp] [--force_wrt_blocks]\n";

GetOptions(
	'show_mt2063' => \$show_mt2063,
	'show_drxk' => \$show_drxk,
	'show_other_reqs' => \$show_other_reqs,
	'show_other_xfer' => \$show_other_xfer,
	'show_other_lines' => \$show_other_lines,
	'show_timestamp' => \$show_timestamp,
	'show_all' => \$show_all,
	'force_wrt_blocks' => \$force_wrt_blocks,
) or die $argerr;

if ($show_all) {
	$show_other_xfer = 1;
	$show_other_lines = 1;
	$show_timestamp = 1;
	$show_other_reqs = 1;
	$show_drxk = 1;
	$show_mt2063 = 1;
}

die ($argerr) if (!($show_other_xfer || $show_other_lines || $show_other_reqs || $show_drxk || $show_mt2063));

sub add_hex_mark($)
{
	my $data = shift;
	my $out ="{";

	return "NULL" if ($data eq "");
	return "NULL due to $data " if ($data =~ /[Ee][Rr][Rr][Oo][Rr]/);

	# Check if the string has the correct syntax. If not, just return it as-is
	for (my $i = 0; $i++; $i < length($data)) {
		if ((($i + 1) % 3) == 0) {
			return $data if (substr($data, $i, 1) ne ' ');
		} else {
			return $data if (!(substr($data, $i, 1) =~ /0-9A-Fa-f/));
		}
	}

	while ($data) {
		$out .= "0x" . substr($data, 0, 2) . ", ";
		$data = substr($data, 3);
	}
	$out =~ s/, $//;
	$out .= "}";

	return $out;
}

my %mt2063_map = (
	0x00 => "MT2063_REG_PART_REV",
	0x01 => "MT2063_REG_LO1CQ_1",
	0x02 => "MT2063_REG_LO1CQ_2",
	0x03 => "MT2063_REG_LO2CQ_1",
	0x04 => "MT2063_REG_LO2CQ_2",
	0x05 => "MT2063_REG_LO2CQ_3",
	0x06 => "MT2063_REG_RSVD_06",
	0x07 => "MT2063_REG_LO_STATUS",
	0x08 => "MT2063_REG_FIFFC",
	0x09 => "MT2063_REG_CLEARTUNE",
	0x0A => "MT2063_REG_ADC_OUT",
	0x0B => "MT2063_REG_LO1C_1",
	0x0C => "MT2063_REG_LO1C_2",
	0x0D => "MT2063_REG_LO2C_1",
	0x0E => "MT2063_REG_LO2C_2",
	0x0F => "MT2063_REG_LO2C_3",
	0x10 => "MT2063_REG_RSVD_10",
	0x11 => "MT2063_REG_PWR_1",
	0x12 => "MT2063_REG_PWR_2",
	0x13 => "MT2063_REG_TEMP_STATUS",
	0x14 => "MT2063_REG_XO_STATUS",
	0x15 => "MT2063_REG_RF_STATUS",
	0x16 => "MT2063_REG_FIF_STATUS",
	0x17 => "MT2063_REG_LNA_OV",
	0x18 => "MT2063_REG_RF_OV",
	0x19 => "MT2063_REG_FIF_OV",
	0x1A => "MT2063_REG_LNA_TGT",
	0x1B => "MT2063_REG_PD1_TGT",
	0x1C => "MT2063_REG_PD2_TGT",
	0x1D => "MT2063_REG_RSVD_1D",
	0x1E => "MT2063_REG_RSVD_1E",
	0x1F => "MT2063_REG_RSVD_1F",
	0x20 => "MT2063_REG_RSVD_20",
	0x21 => "MT2063_REG_BYP_CTRL",
	0x22 => "MT2063_REG_RSVD_22",
	0x23 => "MT2063_REG_RSVD_23",
	0x24 => "MT2063_REG_RSVD_24",
	0x25 => "MT2063_REG_RSVD_25",
	0x26 => "MT2063_REG_RSVD_26",
	0x27 => "MT2063_REG_RSVD_27",
	0x28 => "MT2063_REG_FIFF_CTRL",
	0x29 => "MT2063_REG_FIFF_OFFSET",
	0x2A => "MT2063_REG_CTUNE_CTRL",
	0x2B => "MT2063_REG_CTUNE_OV",
	0x2C => "MT2063_REG_CTRL_2C",
	0x2D => "MT2063_REG_FIFF_CTRL2",
	0x2E => "MT2063_REG_RSVD_2E",
	0x2F => "MT2063_REG_DNC_GAIN",
	0x30 => "MT2063_REG_VGA_GAIN",
	0x31 => "MT2063_REG_RSVD_31",
	0x32 => "MT2063_REG_TEMP_SEL",
	0x33 => "MT2063_REG_RSVD_33",
	0x34 => "MT2063_REG_RSVD_34",
	0x35 => "MT2063_REG_RSVD_35",
	0x36 => "MT2063_REG_RSVD_36",
	0x37 => "MT2063_REG_RSVD_37",
	0x38 => "MT2063_REG_RSVD_38",
	0x39 => "MT2063_REG_RSVD_39",
	0x3A => "MT2063_REG_RSVD_3A",
	0x3B => "MT2063_REG_RSVD_3B",
	0x3C => "MT2063_REG_RSVD_3C",
);

sub parse_mt2063_addr($$$$$)
{
	my $timestamp = shift;
	my $addr = shift;
	my $n = shift;
	my $data = shift;
	my $write = shift;

	my $reg = hex(substr($data, 0, 2));
	$data = substr($data, 3);
	$n--;

	if (defined($mt2063_map{$reg})) {
		$reg = $mt2063_map{$reg};
	} else {
		$reg = sprintf "0x%02x", $reg;
	}

	printf "$timestamp " if ($show_timestamp);
	if ($write) {
		if ($n == 1) {
			printf "mt2063_setreg(state, %s, 0x%s);\n",
				$reg, $data;
		} else {
			printf "mt2063_write(state, %s, %s, %d);\n",
				$reg, add_hex_mark($data), $n;
		}
	} else {
		printf "mt2063_read(state, %s, &buf, %d); /* %s */\n",
			$reg, $n, add_hex_mark($data);
	}
}

# DRX-K registers used at the public driver
# Obtained with this command:
#   cat drivers/media/dvb/frontends/drxk_map.h |perl -ne 'if (/define\s+([^\s]+__A)\s+(.*)/) { printf "\t0x%08x => \"%s\",\n",hex($2), $1; }'
#
my %drxk_map = (
	0x00400000 => "SIO_COMM_EXEC__A",
	0x00400001 => "SIO_COMM_STATE__A",
	0x00400002 => "SIO_COMM_MB__A",
	0x00400003 => "SIO_COMM_INT_REQ__A",
	0x00400005 => "SIO_COMM_INT_STA__A",
	0x00400006 => "SIO_COMM_INT_MSK__A",
	0x00400007 => "SIO_COMM_INT_STM__A",
	0x00410000 => "SIO_TOP_COMM_EXEC__A",
	0x0041000f => "SIO_TOP_COMM_KEY__A",
	0x00410012 => "SIO_TOP_JTAGID_LO__A",
	0x00410013 => "SIO_TOP_JTAGID_HI__A",
	0x00420010 => "SIO_HI_RA_RAM_S0_FLG_SMM__A",
	0x00420011 => "SIO_HI_RA_RAM_S0_DEV_ID__A",
	0x00420012 => "SIO_HI_RA_RAM_S0_FLG_CRC__A",
	0x00420013 => "SIO_HI_RA_RAM_S0_FLG_ACC__A",
	0x00420014 => "SIO_HI_RA_RAM_S0_STATE__A",
	0x00420015 => "SIO_HI_RA_RAM_S0_BLK_BNK__A",
	0x00420016 => "SIO_HI_RA_RAM_S0_ADDR__A",
	0x00420017 => "SIO_HI_RA_RAM_S0_CRC__A",
	0x00420018 => "SIO_HI_RA_RAM_S0_BUFFER__A",
	0x00420019 => "SIO_HI_RA_RAM_S0_RMWBUF__A",
	0x0042001a => "SIO_HI_RA_RAM_S0_FLG_VB__A",
	0x0042001b => "SIO_HI_RA_RAM_S0_TEMP0__A",
	0x0042001c => "SIO_HI_RA_RAM_S0_TEMP1__A",
	0x0042001d => "SIO_HI_RA_RAM_S0_OFFSET__A",
	0x00420020 => "SIO_HI_RA_RAM_S1_FLG_SMM__A",
	0x00420021 => "SIO_HI_RA_RAM_S1_DEV_ID__A",
	0x00420022 => "SIO_HI_RA_RAM_S1_FLG_CRC__A",
	0x00420023 => "SIO_HI_RA_RAM_S1_FLG_ACC__A",
	0x00420024 => "SIO_HI_RA_RAM_S1_STATE__A",
	0x00420025 => "SIO_HI_RA_RAM_S1_BLK_BNK__A",
	0x00420026 => "SIO_HI_RA_RAM_S1_ADDR__A",
	0x00420027 => "SIO_HI_RA_RAM_S1_CRC__A",
	0x00420028 => "SIO_HI_RA_RAM_S1_BUFFER__A",
	0x00420029 => "SIO_HI_RA_RAM_S1_RMWBUF__A",
	0x0042002a => "SIO_HI_RA_RAM_S1_FLG_VB__A",
	0x0042002b => "SIO_HI_RA_RAM_S1_TEMP0__A",
	0x0042002c => "SIO_HI_RA_RAM_S1_TEMP1__A",
	0x0042002d => "SIO_HI_RA_RAM_S1_OFFSET__A",
	0x00420030 => "SIO_HI_RA_RAM_SEMA__A",
	0x00420031 => "SIO_HI_RA_RAM_RES__A",
	0x00420032 => "SIO_HI_RA_RAM_CMD__A",
	0x00420033 => "SIO_HI_RA_RAM_PAR_1__A",
	0x00420034 => "SIO_HI_RA_RAM_PAR_2__A",
	0x00420035 => "SIO_HI_RA_RAM_PAR_3__A",
	0x00420036 => "SIO_HI_RA_RAM_PAR_4__A",
	0x00420037 => "SIO_HI_RA_RAM_PAR_5__A",
	0x00420038 => "SIO_HI_RA_RAM_PAR_6__A",
	0x0042006e => "SIO_HI_RA_RAM_AB_TEMP__A",
	0x0042006f => "SIO_HI_RA_RAM_I2C_CTL__A",
	0x00420070 => "SIO_HI_RA_RAM_VB_ENTRY0__A",
	0x00420071 => "SIO_HI_RA_RAM_VB_OFFSET0__A",
	0x00420072 => "SIO_HI_RA_RAM_VB_ENTRY1__A",
	0x00420073 => "SIO_HI_RA_RAM_VB_OFFSET1__A",
	0x00420074 => "SIO_HI_RA_RAM_VB_ENTRY2__A",
	0x00420075 => "SIO_HI_RA_RAM_VB_OFFSET2__A",
	0x00420076 => "SIO_HI_RA_RAM_VB_ENTRY3__A",
	0x00420077 => "SIO_HI_RA_RAM_VB_OFFSET3__A",
	0x00420078 => "SIO_HI_RA_RAM_VB_ENTRY4__A",
	0x00420079 => "SIO_HI_RA_RAM_VB_OFFSET4__A",
	0x0042007a => "SIO_HI_RA_RAM_VB_ENTRY5__A",
	0x0042007b => "SIO_HI_RA_RAM_VB_OFFSET5__A",
	0x0042007c => "SIO_HI_RA_RAM_VB_ENTRY6__A",
	0x0042007d => "SIO_HI_RA_RAM_VB_OFFSET6__A",
	0x0042007e => "SIO_HI_RA_RAM_VB_ENTRY7__A",
	0x0042007f => "SIO_HI_RA_RAM_VB_OFFSET7__A",
	0x00430000 => "SIO_HI_IF_RAM_TRP_BPT_0__A",
	0x00430001 => "SIO_HI_IF_RAM_TRP_BPT_1__A",
	0x00430002 => "SIO_HI_IF_RAM_TRP_STK_0__A",
	0x00430003 => "SIO_HI_IF_RAM_TRP_STK_1__A",
	0x00430300 => "SIO_HI_IF_RAM_FUN_BASE__A",
	0x00440000 => "SIO_HI_IF_COMM_EXEC__A",
	0x00440001 => "SIO_HI_IF_COMM_STATE__A",
	0x00440003 => "SIO_HI_IF_COMM_INT_REQ__A",
	0x00440005 => "SIO_HI_IF_COMM_INT_STA__A",
	0x00440006 => "SIO_HI_IF_COMM_INT_MSK__A",
	0x00440007 => "SIO_HI_IF_COMM_INT_STM__A",
	0x00440010 => "SIO_HI_IF_STK_0__A",
	0x00440011 => "SIO_HI_IF_STK_1__A",
	0x00440012 => "SIO_HI_IF_STK_2__A",
	0x00440013 => "SIO_HI_IF_STK_3__A",
	0x0044001f => "SIO_HI_IF_BPT_IDX__A",
	0x00440020 => "SIO_HI_IF_BPT__A",
	0x00450000 => "SIO_CC_COMM_EXEC__A",
	0x00450010 => "SIO_CC_PLL_MODE__A",
	0x00450011 => "SIO_CC_PLL_TEST__A",
	0x00450012 => "SIO_CC_PLL_LOCK__A",
	0x00450013 => "SIO_CC_CLK_TEST__A",
	0x00450014 => "SIO_CC_CLK_MODE__A",
	0x00450015 => "SIO_CC_PWD_MODE__A",
	0x00450016 => "SIO_CC_SOFT_RST__A",
	0x00450017 => "SIO_CC_UPDATE__A",
	0x00460000 => "SIO_SA_COMM_EXEC__A",
	0x00460003 => "SIO_SA_COMM_INT_REQ__A",
	0x00460005 => "SIO_SA_COMM_INT_STA__A",
	0x00460006 => "SIO_SA_COMM_INT_MSK__A",
	0x00460007 => "SIO_SA_COMM_INT_STM__A",
	0x00460010 => "SIO_SA_PRESCALER__A",
	0x00460011 => "SIO_SA_TX_DATA0__A",
	0x00460012 => "SIO_SA_TX_DATA1__A",
	0x00460013 => "SIO_SA_TX_DATA2__A",
	0x00460014 => "SIO_SA_TX_DATA3__A",
	0x00460015 => "SIO_SA_TX_LENGTH__A",
	0x00460016 => "SIO_SA_TX_COMMAND__A",
	0x00460017 => "SIO_SA_TX_STATUS__A",
	0x00460018 => "SIO_SA_RX_DATA0__A",
	0x00460019 => "SIO_SA_RX_DATA1__A",
	0x0046001a => "SIO_SA_RX_LENGTH__A",
	0x0046001b => "SIO_SA_RX_COMMAND__A",
	0x0046001c => "SIO_SA_RX_STATUS__A",
	0x00470000 => "SIO_OFDM_SH_COMM_EXEC__A",
	0x00470002 => "SIO_OFDM_SH_COMM_MB__A",
	0x00470010 => "SIO_OFDM_SH_OFDM_RING_ENABLE__A",
	0x00470011 => "SIO_OFDM_SH_OFDM_MB_CONTROL__A",
	0x00470012 => "SIO_OFDM_SH_OFDM_RING_STATUS__A",
	0x00470013 => "SIO_OFDM_SH_OFDM_MB_FLEN__A",
	0x00480000 => "SIO_BL_COMM_EXEC__A",
	0x00480003 => "SIO_BL_COMM_INT_REQ__A",
	0x00480005 => "SIO_BL_COMM_INT_STA__A",
	0x00480006 => "SIO_BL_COMM_INT_MSK__A",
	0x00480007 => "SIO_BL_COMM_INT_STM__A",
	0x00480010 => "SIO_BL_STATUS__A",
	0x00480011 => "SIO_BL_MODE__A",
	0x00480012 => "SIO_BL_ENABLE__A",
	0x00480014 => "SIO_BL_TGT_HDR__A",
	0x00480015 => "SIO_BL_TGT_ADDR__A",
	0x00480016 => "SIO_BL_SRC_ADDR__A",
	0x00480017 => "SIO_BL_SRC_LEN__A",
	0x00480018 => "SIO_BL_CHAIN_ADDR__A",
	0x00480019 => "SIO_BL_CHAIN_LEN__A",
	0x004c0000 => "SIO_OFDM_SH_TRB_R0_RAM__A",
	0x004d0000 => "SIO_OFDM_SH_TRB_R1_RAM__A",
	0x004e0000 => "SIO_BL_ROM__A",
	0x007f0000 => "SIO_PDR_COMM_EXEC__A",
	0x007f0010 => "SIO_PDR_MON_CFG__A",
	0x007f0012 => "SIO_PDR_SMA_RX_SEL__A",
	0x007f0013 => "SIO_PDR_SILENT__A",
	0x007f0014 => "SIO_PDR_UIO_IN_LO__A",
	0x007f0015 => "SIO_PDR_UIO_IN_HI__A",
	0x007f0016 => "SIO_PDR_UIO_OUT_LO__A",
	0x007f0017 => "SIO_PDR_UIO_OUT_HI__A",
	0x007f0018 => "SIO_PDR_PWM1_MODE__A",
	0x007f0019 => "SIO_PDR_PWM1_PRESCALE__A",
	0x007f001a => "SIO_PDR_PWM1_VALUE__A",
	0x007f001b => "SIO_PDR_IRQN_SEL__A",
	0x007f001c => "SIO_PDR_PWM2_MODE__A",
	0x007f001d => "SIO_PDR_PWM2_PRESCALE__A",
	0x007f001e => "SIO_PDR_PWM2_VALUE__A",
	0x007f001f => "SIO_PDR_OHW_CFG__A",
	0x007f0020 => "SIO_PDR_I2S_WS_CFG__A",
	0x007f0021 => "SIO_PDR_GPIO_CFG__A",
	0x007f0025 => "SIO_PDR_MSTRT_CFG__A",
	0x007f0026 => "SIO_PDR_MERR_CFG__A",
	0x007f0028 => "SIO_PDR_MCLK_CFG__A",
	0x007f0029 => "SIO_PDR_MVAL_CFG__A",
	0x007f002a => "SIO_PDR_MD0_CFG__A",
	0x007f002b => "SIO_PDR_MD1_CFG__A",
	0x007f002c => "SIO_PDR_MD2_CFG__A",
	0x007f002d => "SIO_PDR_MD3_CFG__A",
	0x007f002f => "SIO_PDR_MD4_CFG__A",
	0x007f0030 => "SIO_PDR_MD5_CFG__A",
	0x007f0031 => "SIO_PDR_MD6_CFG__A",
	0x007f0032 => "SIO_PDR_MD7_CFG__A",
	0x007f0033 => "SIO_PDR_I2C_SCL1_CFG__A",
	0x007f0034 => "SIO_PDR_I2C_SDA1_CFG__A",
	0x007f0036 => "SIO_PDR_VSYNC_CFG__A",
	0x007f0037 => "SIO_PDR_SMA_RX_CFG__A",
	0x007f0038 => "SIO_PDR_SMA_TX_CFG__A",
	0x007f003f => "SIO_PDR_I2C_SDA2_CFG__A",
	0x007f0040 => "SIO_PDR_I2C_SCL2_CFG__A",
	0x007f0041 => "SIO_PDR_I2S_CL_CFG__A",
	0x007f0042 => "SIO_PDR_I2S_DA_CFG__A",
	0x007f0050 => "SIO_PDR_GPIO_GPIO_FNC__A",
	0x007f0052 => "SIO_PDR_MSTRT_GPIO_FNC__A",
	0x007f0053 => "SIO_PDR_MERR_GPIO_FNC__A",
	0x007f0054 => "SIO_PDR_MCLK_GPIO_FNC__A",
	0x007f0055 => "SIO_PDR_MVAL_GPIO_FNC__A",
	0x007f0056 => "SIO_PDR_MD0_GPIO_FNC__A",
	0x007f0057 => "SIO_PDR_MD1_GPIO_FNC__A",
	0x007f0058 => "SIO_PDR_MD2_GPIO_FNC__A",
	0x007f0059 => "SIO_PDR_MD3_GPIO_FNC__A",
	0x007f005a => "SIO_PDR_MD4_GPIO_FNC__A",
	0x007f005b => "SIO_PDR_MD5_GPIO_FNC__A",
	0x007f005c => "SIO_PDR_MD6_GPIO_FNC__A",
	0x007f005d => "SIO_PDR_MD7_GPIO_FNC__A",
	0x007f005e => "SIO_PDR_SMA_RX_GPIO_FNC__A",
	0x007f005f => "SIO_PDR_SMA_TX_GPIO_FNC__A",
	0x00800000 => "SCU_COMM_EXEC__A",
	0x00800001 => "SCU_COMM_STATE__A",
	0x00810000 => "SCU_TOP_COMM_EXEC__A",
	0x00810001 => "SCU_TOP_COMM_STATE__A",
	0x00810010 => "SCU_TOP_MWAIT_CTR__A",
	0x00820000 => "SCU_LOW_RAM__A",
	0x00830000 => "SCU_HIGH_RAM__A",
	0x00831ebf => "SCU_RAM_DRIVER_DEBUG__A",
	0x00831ec0 => "SCU_RAM_SP__A",
	0x00831ec1 => "SCU_RAM_QAM_NEVERLOCK_CNT__A",
	0x00831ec2 => "SCU_RAM_QAM_WRONG_RATE_CNT__A",
	0x00831ec3 => "SCU_RAM_QAM_NO_ACQ_CNT__A",
	0x00831ec4 => "SCU_RAM_QAM_FSM_STEP_PERIOD__A",
	0x00831ec5 => "SCU_RAM_AGC_KI_MIN_IFGAIN__A",
	0x00831ec6 => "SCU_RAM_AGC_KI_MAX_IFGAIN__A",
	0x00831ec7 => "SCU_RAM_GPIO__A",
	0x00831ec8 => "SCU_RAM_AGC_CLP_CTRL_MODE__A",
	0x00831ec9 => "SCU_RAM_AGC_KI_MIN_RFGAIN__A",
	0x00831eca => "SCU_RAM_AGC_KI_MAX_RFGAIN__A",
	0x00831ecb => "SCU_RAM_FEC_ACCUM_PKT_FAILURES__A",
	0x00831ecc => "SCU_RAM_INHIBIT_1__A",
	0x00831ecd => "SCU_RAM_HTOL_BUF_0__A",
	0x00831ece => "SCU_RAM_HTOL_BUF_1__A",
	0x00831ecf => "SCU_RAM_INHIBIT_2__A",
	0x00831ed0 => "SCU_RAM_TR_SHORT_BUF_0__A",
	0x00831ed1 => "SCU_RAM_TR_SHORT_BUF_1__A",
	0x00831ed2 => "SCU_RAM_TR_LONG_BUF_0__A",
	0x00831ed3 => "SCU_RAM_TR_LONG_BUF_1__A",
	0x00831ed4 => "SCU_RAM_TR_LONG_BUF_2__A",
	0x00831ed5 => "SCU_RAM_TR_LONG_BUF_3__A",
	0x00831ed6 => "SCU_RAM_TR_LONG_BUF_4__A",
	0x00831ed7 => "SCU_RAM_TR_LONG_BUF_5__A",
	0x00831ed8 => "SCU_RAM_TR_LONG_BUF_6__A",
	0x00831ed9 => "SCU_RAM_TR_LONG_BUF_7__A",
	0x00831eda => "SCU_RAM_TR_LONG_BUF_8__A",
	0x00831edb => "SCU_RAM_TR_LONG_BUF_9__A",
	0x00831edc => "SCU_RAM_TR_LONG_BUF_10__A",
	0x00831edd => "SCU_RAM_TR_LONG_BUF_11__A",
	0x00831ede => "SCU_RAM_TR_LONG_BUF_12__A",
	0x00831edf => "SCU_RAM_TR_LONG_BUF_13__A",
	0x00831ee0 => "SCU_RAM_TR_LONG_BUF_14__A",
	0x00831ee1 => "SCU_RAM_TR_LONG_BUF_15__A",
	0x00831ee2 => "SCU_RAM_TR_LONG_BUF_16__A",
	0x00831ee3 => "SCU_RAM_TR_LONG_BUF_17__A",
	0x00831ee4 => "SCU_RAM_TR_LONG_BUF_18__A",
	0x00831ee5 => "SCU_RAM_TR_LONG_BUF_19__A",
	0x00831ee6 => "SCU_RAM_TR_LONG_BUF_20__A",
	0x00831ee7 => "SCU_RAM_TR_LONG_BUF_21__A",
	0x00831ee8 => "SCU_RAM_TR_LONG_BUF_22__A",
	0x00831ee9 => "SCU_RAM_TR_LONG_BUF_23__A",
	0x00831eea => "SCU_RAM_TR_LONG_BUF_24__A",
	0x00831eeb => "SCU_RAM_TR_LONG_BUF_25__A",
	0x00831eec => "SCU_RAM_TR_LONG_BUF_26__A",
	0x00831eed => "SCU_RAM_TR_LONG_BUF_27__A",
	0x00831eee => "SCU_RAM_TR_LONG_BUF_28__A",
	0x00831eef => "SCU_RAM_TR_LONG_BUF_29__A",
	0x00831ef0 => "SCU_RAM_TR_LONG_BUF_30__A",
	0x00831ef1 => "SCU_RAM_TR_LONG_BUF_31__A",
	0x00831ef2 => "SCU_RAM_ATV_AMS_MAX__A",
	0x00831ef3 => "SCU_RAM_ATV_AMS_MIN__A",
	0x00831ef4 => "SCU_RAM_ATV_FIELD_CNT__A",
	0x00831ef5 => "SCU_RAM_ATV_AAGC_FAST__A",
	0x00831ef6 => "SCU_RAM_ATV_AAGC_LP2__A",
	0x00831ef7 => "SCU_RAM_ATV_BP_LVL__A",
	0x00831ef8 => "SCU_RAM_ATV_BP_RELY__A",
	0x00831ef9 => "SCU_RAM_ATV_BP_MTA__A",
	0x00831efa => "SCU_RAM_ATV_BP_REF__A",
	0x00831efb => "SCU_RAM_ATV_BP_REF_MIN__A",
	0x00831efc => "SCU_RAM_ATV_BP_REF_MAX__A",
	0x00831efd => "SCU_RAM_ATV_BP_CNT__A",
	0x00831efe => "SCU_RAM_ATV_BP_XD_CNT__A",
	0x00831eff => "SCU_RAM_ATV_PAGC_KI_MIN__A",
	0x00831f00 => "SCU_RAM_ATV_BPC_KI_MIN__A",
	0x00831f01 => "SCU_RAM_OFDM_AGC_POW_TGT__A",
	0x00831f02 => "SCU_RAM_OFDM_RSV_01__A",
	0x00831f03 => "SCU_RAM_OFDM_RSV_02__A",
	0x00831f04 => "SCU_RAM_FEC_PRE_RS_BER__A",
	0x00831f05 => "SCU_RAM_FEC_PRE_RS_BER_FILTER_SH__A",
	0x00831f06 => "SCU_RAM_FREE_7942__A",
	0x00831f07 => "SCU_RAM_FREE_7943__A",
	0x00831f08 => "SCU_RAM_FREE_7944__A",
	0x00831f09 => "SCU_RAM_FREE_7945__A",
	0x00831f0a => "SCU_RAM_FREE_7946__A",
	0x00831f0b => "SCU_RAM_FREE_7947__A",
	0x00831f0c => "SCU_RAM_FREE_7948__A",
	0x00831f0d => "SCU_RAM_FREE_7949__A",
	0x00831f0e => "SCU_RAM_FREE_7950__A",
	0x00831f0f => "SCU_RAM_FREE_7951__A",
	0x00831f10 => "SCU_RAM_FREE_7952__A",
	0x00831f11 => "SCU_RAM_FREE_7953__A",
	0x00831f12 => "SCU_RAM_FREE_7954__A",
	0x00831f13 => "SCU_RAM_FREE_7955__A",
	0x00831f14 => "SCU_RAM_ADC_COMP_CONTROL__A",
	0x00831f15 => "SCU_RAM_AGC_FAST_SNS_CTRL_DELAY__A",
	0x00831f16 => "SCU_RAM_AGC_KI_CYCCNT__A",
	0x00831f17 => "SCU_RAM_AGC_KI_CYCLEN__A",
	0x00831f18 => "SCU_RAM_AGC_SNS_CYCLEN__A",
	0x00831f19 => "SCU_RAM_AGC_RF_SNS_DEV_MAX__A",
	0x00831f1a => "SCU_RAM_AGC_RF_SNS_DEV_MIN__A",
	0x00831f1b => "SCU_RAM_AGC_RF_MAX__A",
	0x00831f1c => "SCU_RAM_FREE_7964__A",
	0x00831f1d => "SCU_RAM_FREE_7965__A",
	0x00831f1e => "SCU_RAM_FREE_7966__A",
	0x00831f1f => "SCU_RAM_FREE_7967__A",
	0x00831f20 => "SCU_RAM_FREE_7968__A",
	0x00831f21 => "SCU_RAM_FREE_7969__A",
	0x00831f22 => "SCU_RAM_FREE_7970__A",
	0x00831f23 => "SCU_RAM_FREE_7971__A",
	0x00831f24 => "SCU_RAM_AGC_CONFIG__A",
	0x00831f24 => "SCU_RAM_FREE_7972__A",
	0x00831f25 => "SCU_RAM_AGC_KI__A",
	0x00831f26 => "SCU_RAM_AGC_KI_RED__A",
	0x00831f27 => "SCU_RAM_AGC_KI_INNERGAIN_MIN__A",
	0x00831f28 => "SCU_RAM_AGC_KI_MINGAIN__A",
	0x00831f29 => "SCU_RAM_AGC_KI_MAXGAIN__A",
	0x00831f2a => "SCU_RAM_AGC_KI_MAXMINGAIN_TH__A",
	0x00831f2b => "SCU_RAM_AGC_KI_MIN__A",
	0x00831f2c => "SCU_RAM_AGC_KI_MAX__A",
	0x00831f2d => "SCU_RAM_AGC_CLP_SUM__A",
	0x00831f2e => "SCU_RAM_AGC_CLP_SUM_MIN__A",
	0x00831f2f => "SCU_RAM_AGC_CLP_SUM_MAX__A",
	0x00831f30 => "SCU_RAM_AGC_CLP_CYCLEN__A",
	0x00831f31 => "SCU_RAM_AGC_CLP_CYCCNT__A",
	0x00831f32 => "SCU_RAM_AGC_CLP_DIR_TO__A",
	0x00831f33 => "SCU_RAM_AGC_CLP_DIR_WD__A",
	0x00831f34 => "SCU_RAM_AGC_CLP_DIR_STP__A",
	0x00831f35 => "SCU_RAM_AGC_SNS_SUM__A",
	0x00831f36 => "SCU_RAM_AGC_SNS_SUM_MIN__A",
	0x00831f37 => "SCU_RAM_AGC_SNS_SUM_MAX__A",
	0x00831f38 => "SCU_RAM_AGC_SNS_CYCCNT__A",
	0x00831f39 => "SCU_RAM_AGC_SNS_DIR_TO__A",
	0x00831f3a => "SCU_RAM_AGC_SNS_DIR_WD__A",
	0x00831f3b => "SCU_RAM_AGC_SNS_DIR_STP__A",
	0x00831f3c => "SCU_RAM_AGC_INGAIN__A",
	0x00831f3d => "SCU_RAM_AGC_INGAIN_TGT__A",
	0x00831f3e => "SCU_RAM_AGC_INGAIN_TGT_MIN__A",
	0x00831f3f => "SCU_RAM_AGC_INGAIN_TGT_MAX__A",
	0x00831f40 => "SCU_RAM_AGC_IF_IACCU_HI__A",
	0x00831f41 => "SCU_RAM_AGC_IF_IACCU_LO__A",
	0x00831f42 => "SCU_RAM_AGC_IF_IACCU_HI_TGT__A",
	0x00831f43 => "SCU_RAM_AGC_IF_IACCU_HI_TGT_MIN__A",
	0x00831f44 => "SCU_RAM_AGC_IF_IACCU_HI_TGT_MAX__A",
	0x00831f45 => "SCU_RAM_AGC_RF_IACCU_HI__A",
	0x00831f46 => "SCU_RAM_AGC_RF_IACCU_LO__A",
	0x00831f47 => "SCU_RAM_AGC_RF_IACCU_HI_CO__A",
	0x00831f48 => "SCU_RAM_ATV_STANDARD__A",
	0x00831f49 => "SCU_RAM_ATV_DETECT__A",
	0x00831f4a => "SCU_RAM_ATV_DETECT_TH__A",
	0x00831f4b => "SCU_RAM_ATV_LOCK__A",
	0x00831f4c => "SCU_RAM_ATV_CR_LOCK__A",
	0x00831f4d => "SCU_RAM_ATV_AGC_MODE__A",
	0x00831f4e => "SCU_RAM_ATV_RSV_01__A",
	0x00831f4f => "SCU_RAM_ATV_RSV_02__A",
	0x00831f50 => "SCU_RAM_ATV_RSV_03__A",
	0x00831f51 => "SCU_RAM_ATV_RSV_04__A",
	0x00831f52 => "SCU_RAM_ATV_FAGC_TH_RED__A",
	0x00831f53 => "SCU_RAM_ATV_AMS_MAX_REF__A",
	0x00831f54 => "SCU_RAM_ATV_ACT_AMX__A",
	0x00831f55 => "SCU_RAM_ATV_ACT_AMI__A",
	0x00831f56 => "SCU_RAM_ATV_BPC_REF_PERIOD__A",
	0x00831f57 => "SCU_RAM_ATV_BPC_REF_CNT__A",
	0x00831f58 => "SCU_RAM_ATV_RSV_07__A",
	0x00831f59 => "SCU_RAM_ATV_RSV_08__A",
	0x00831f5a => "SCU_RAM_ATV_RSV_09__A",
	0x00831f5b => "SCU_RAM_ATV_RSV_10__A",
	0x00831f5c => "SCU_RAM_ATV_RSV_11__A",
	0x00831f5d => "SCU_RAM_ATV_RSV_12__A",
	0x00831f5e => "SCU_RAM_ATV_VID_GAIN_HI__A",
	0x00831f5f => "SCU_RAM_ATV_VID_GAIN_LO__A",
	0x00831f60 => "SCU_RAM_ATV_RSV_13__A",
	0x00831f61 => "SCU_RAM_ATV_RSV_14__A",
	0x00831f62 => "SCU_RAM_ATV_RSV_15__A",
	0x00831f63 => "SCU_RAM_ATV_RSV_16__A",
	0x00831f64 => "SCU_RAM_ATV_AAGC_CNT__A",
	0x00831f65 => "SCU_RAM_ATV_SIF_GAIN__A",
	0x00831f66 => "SCU_RAM_ATV_RSV_17__A",
	0x00831f67 => "SCU_RAM_ATV_RSV_18__A",
	0x00831f68 => "SCU_RAM_ATV_RATE_OFS__A",
	0x00831f69 => "SCU_RAM_ATV_LO_INCR__A",
	0x00831f6a => "SCU_RAM_ATV_IIR_CRIT__A",
	0x00831f6b => "SCU_RAM_ATV_DEF_RATE_OFS__A",
	0x00831f6c => "SCU_RAM_ATV_DEF_LO_INCR__A",
	0x00831f6d => "SCU_RAM_ATV_ENABLE_IIR_WA__A",
	0x00831f6e => "SCU_RAM_ATV_MOD_CONTROL__A",
	0x00831f6f => "SCU_RAM_ATV_PAGC_KI_MAX__A",
	0x00831f70 => "SCU_RAM_ATV_BPC_KI_MAX__A",
	0x00831f71 => "SCU_RAM_ATV_NAGC_KI_MAX__A",
	0x00831f72 => "SCU_RAM_ATV_NAGC_KI_MIN__A",
	0x00831f73 => "SCU_RAM_ATV_KI_CHANGE_TH__A",
	0x00831f74 => "SCU_RAM_QAM_PARAM_ANNEX__A",
	0x00831f75 => "SCU_RAM_QAM_PARAM_CONSTELLATION__A",
	0x00831f76 => "SCU_RAM_QAM_PARAM_INTERLEAVE__A",
	0x00831f77 => "SCU_RAM_QAM_PARAM_SYM_RCRATE_HI__A",
	0x00831f78 => "SCU_RAM_QAM_PARAM_SYM_RCRATE_LO__A",
	0x00831f79 => "SCU_RAM_QAM_EQ_CENTERTAP__A",
	0x00831f7a => "SCU_RAM_QAM_WR_RSV_0__A",
	0x00831f7b => "SCU_RAM_QAM_PARAM_ALT_RCRATE_HI__A",
	0x00831f7c => "SCU_RAM_QAM_PARAM_ALT_RCRATE_LO__A",
	0x00831f7d => "SCU_RAM_QAM_WR_RSV_5__A",
	0x00831f7e => "SCU_RAM_QAM_WR_RSV_6__A",
	0x00831f7f => "SCU_RAM_QAM_WR_RSV_7__A",
	0x00831f80 => "SCU_RAM_QAM_WR_RSV_8__A",
	0x00831f81 => "SCU_RAM_QAM_WR_RSV_9__A",
	0x00831f82 => "SCU_RAM_QAM_WR_RSV_10__A",
	0x00831f83 => "SCU_RAM_QAM_FSM_FMHUM_TO__A",
	0x00831f84 => "SCU_RAM_QAM_FSM_MEDIAN_AV_MULT__A",
	0x00831f85 => "SCU_RAM_QAM_FSM_RADIUS_AV_LIMIT__A",
	0x00831f86 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET1__A",
	0x00831f87 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET2__A",
	0x00831f88 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET3__A",
	0x00831f89 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET4__A",
	0x00831f8a => "SCU_RAM_QAM_FSM_LCAVG_OFFSET5__A",
	0x00831f8b => "SCU_RAM_QAM_FSM_STATE_TGT__A",
	0x00831f8c => "SCU_RAM_QAM_FSM_LOCK_OVERRIDE__A",
	0x00831f8d => "SCU_RAM_QAM_FSM_ATH__A",
	0x00831f8e => "SCU_RAM_QAM_FSM_RTH__A",
	0x00831f8f => "SCU_RAM_QAM_FSM_FTH__A",
	0x00831f90 => "SCU_RAM_QAM_FSM_PTH__A",
	0x00831f91 => "SCU_RAM_QAM_FSM_MTH__A",
	0x00831f92 => "SCU_RAM_QAM_FSM_CTH__A",
	0x00831f93 => "SCU_RAM_QAM_FSM_QTH__A",
	0x00831f94 => "SCU_RAM_QAM_FSM_RATE_LIM__A",
	0x00831f95 => "SCU_RAM_QAM_FSM_FREQ_LIM__A",
	0x00831f96 => "SCU_RAM_QAM_FSM_COUNT_LIM__A",
	0x00831f97 => "SCU_RAM_QAM_LC_CA_COARSE__A",
	0x00831f98 => "SCU_RAM_QAM_LC_CA_MEDIUM__A",
	0x00831f99 => "SCU_RAM_QAM_LC_CA_FINE__A",
	0x00831f9a => "SCU_RAM_QAM_LC_CP_COARSE__A",
	0x00831f9b => "SCU_RAM_QAM_LC_CP_MEDIUM__A",
	0x00831f9c => "SCU_RAM_QAM_LC_CP_FINE__A",
	0x00831f9d => "SCU_RAM_QAM_LC_CI_COARSE__A",
	0x00831f9e => "SCU_RAM_QAM_LC_CI_MEDIUM__A",
	0x00831f9f => "SCU_RAM_QAM_LC_CI_FINE__A",
	0x00831fa0 => "SCU_RAM_QAM_LC_EP_COARSE__A",
	0x00831fa1 => "SCU_RAM_QAM_LC_EP_MEDIUM__A",
	0x00831fa2 => "SCU_RAM_QAM_LC_EP_FINE__A",
	0x00831fa3 => "SCU_RAM_QAM_LC_EI_COARSE__A",
	0x00831fa4 => "SCU_RAM_QAM_LC_EI_MEDIUM__A",
	0x00831fa5 => "SCU_RAM_QAM_LC_EI_FINE__A",
	0x00831fa6 => "SCU_RAM_QAM_LC_CF_COARSE__A",
	0x00831fa7 => "SCU_RAM_QAM_LC_CF_MEDIUM__A",
	0x00831fa8 => "SCU_RAM_QAM_LC_CF_FINE__A",
	0x00831fa9 => "SCU_RAM_QAM_LC_CF1_COARSE__A",
	0x00831faa => "SCU_RAM_QAM_LC_CF1_MEDIUM__A",
	0x00831fab => "SCU_RAM_QAM_LC_CF1_FINE__A",
	0x00831fac => "SCU_RAM_QAM_SL_SIG_POWER__A",
	0x00831fad => "SCU_RAM_QAM_EQ_CMA_RAD0__A",
	0x00831fae => "SCU_RAM_QAM_EQ_CMA_RAD1__A",
	0x00831faf => "SCU_RAM_QAM_EQ_CMA_RAD2__A",
	0x00831fb0 => "SCU_RAM_QAM_EQ_CMA_RAD3__A",
	0x00831fb1 => "SCU_RAM_QAM_EQ_CMA_RAD4__A",
	0x00831fb2 => "SCU_RAM_QAM_EQ_CMA_RAD5__A",
	0x00831fb3 => "SCU_RAM_QAM_CTL_ENA__A",
	0x00831fb4 => "SCU_RAM_QAM_WR_RSV_1__A",
	0x00831fb5 => "SCU_RAM_QAM_WR_RSV_2__A",
	0x00831fb6 => "SCU_RAM_QAM_WR_RSV_3__A",
	0x00831fb7 => "SCU_RAM_QAM_ACTIVE_CONSTELLATION__A",
	0x00831fb8 => "SCU_RAM_QAM_ACTIVE_INTERLEAVE__A",
	0x00831fb9 => "SCU_RAM_QAM_RD_RSV_4__A",
	0x00831fba => "SCU_RAM_QAM_LOCKED__A",
	0x00831fbb => "SCU_RAM_QAM_EVENTS_OCC_HI__A",
	0x00831fbc => "SCU_RAM_QAM_EVENTS_OCC_LO__A",
	0x00831fbd => "SCU_RAM_QAM_EVENTS_SCHED_HI__A",
	0x00831fbe => "SCU_RAM_QAM_EVENTS_SCHED_LO__A",
	0x00831fbf => "SCU_RAM_QAM_TASKLETS_SCHED__A",
	0x00831fc0 => "SCU_RAM_QAM_TASKLETS_RUN__A",
	0x00831fc1 => "SCU_RAM_QAM_ACTIVE_SYM_RCRATE_HI__A",
	0x00831fc2 => "SCU_RAM_QAM_ACTIVE_SYM_RCRATE_LO__A",
	0x00831fc3 => "SCU_RAM_QAM_RD_RSV_5__A",
	0x00831fc4 => "SCU_RAM_QAM_RD_RSV_6__A",
	0x00831fc5 => "SCU_RAM_QAM_RD_RSV_7__A",
	0x00831fc6 => "SCU_RAM_QAM_RD_RSV_8__A",
	0x00831fc7 => "SCU_RAM_QAM_RD_RSV_9__A",
	0x00831fc8 => "SCU_RAM_QAM_RD_RSV_10__A",
	0x00831fc9 => "SCU_RAM_QAM_AGC_TPOW_OFFS__A",
	0x00831fca => "SCU_RAM_QAM_FSM_STATE__A",
	0x00831fcb => "SCU_RAM_QAM_FSM_STATE_NEW__A",
	0x00831fcc => "SCU_RAM_QAM_FSM_LOCK_FLAGS__A",
	0x00831fcd => "SCU_RAM_QAM_FSM_RATE_VARIATION__A",
	0x00831fce => "SCU_RAM_QAM_FSM_FREQ_VARIATION__A",
	0x00831fcf => "SCU_RAM_QAM_ERR_STATE__A",
	0x00831fd0 => "SCU_RAM_QAM_ERR_LOCK_FLAGS__A",
	0x00831fd1 => "SCU_RAM_QAM_EQ_LOCK__A",
	0x00831fd2 => "SCU_RAM_QAM_EQ_STATE__A",
	0x00831fd3 => "SCU_RAM_QAM_RD_RSV_0__A",
	0x00831fd4 => "SCU_RAM_QAM_RD_RSV_1__A",
	0x00831fd5 => "SCU_RAM_QAM_RD_RSV_2__A",
	0x00831fd6 => "SCU_RAM_QAM_RD_RSV_3__A",
	0x00831fd7 => "SCU_RAM_FREE_8151__A",
	0x00831fd8 => "SCU_RAM_FREE_8152__A",
	0x00831fd9 => "SCU_RAM_FREE_8153__A",
	0x00831fda => "SCU_RAM_FREE_8154__A",
	0x00831fdb => "SCU_RAM_FREE_8155__A",
	0x00831fdc => "SCU_RAM_FREE_8156__A",
	0x00831fdd => "SCU_RAM_FREE_8157__A",
	0x00831fde => "SCU_RAM_FREE_8158__A",
	0x00831fdf => "SCU_RAM_FREE_8159__A",
	0x00831fe0 => "SCU_RAM_FREE_8160__A",
	0x00831fe1 => "SCU_RAM_FREE_8161__A",
	0x00831fe2 => "SCU_RAM_FREE_8162__A",
	0x00831fe3 => "SCU_RAM_FREE_8163__A",
	0x00831fe4 => "SCU_RAM_FREE_8164__A",
	0x00831fe5 => "SCU_RAM_FREE_8165__A",
	0x00831fe6 => "SCU_RAM_FREE_8166__A",
	0x00831fe7 => "SCU_RAM_FREE_8167__A",
	0x00831fe8 => "SCU_RAM_FREE_8168__A",
	0x00831fe9 => "SCU_RAM_FREE_8169__A",
	0x00831fea => "SCU_RAM_AGC_FAST_CLP_CTRL_DELAY__A",
	0x00831feb => "SCU_RAM_DRIVER_VER_HI__A",
	0x00831fec => "SCU_RAM_DRIVER_VER_LO__A",
	0x00831fed => "SCU_RAM_PARAM_15__A",
	0x00831fee => "SCU_RAM_PARAM_14__A",
	0x00831fef => "SCU_RAM_PARAM_13__A",
	0x00831ff0 => "SCU_RAM_PARAM_12__A",
	0x00831ff1 => "SCU_RAM_PARAM_11__A",
	0x00831ff2 => "SCU_RAM_PARAM_10__A",
	0x00831ff3 => "SCU_RAM_PARAM_9__A",
	0x00831ff4 => "SCU_RAM_PARAM_8__A",
	0x00831ff5 => "SCU_RAM_PARAM_7__A",
	0x00831ff6 => "SCU_RAM_PARAM_6__A",
	0x00831ff7 => "SCU_RAM_PARAM_5__A",
	0x00831ff8 => "SCU_RAM_PARAM_4__A",
	0x00831ff9 => "SCU_RAM_PARAM_3__A",
	0x00831ffa => "SCU_RAM_PARAM_2__A",
	0x00831ffb => "SCU_RAM_PARAM_1__A",
	0x00831ffc => "SCU_RAM_PARAM_0__A",
	0x00831ffd => "SCU_RAM_COMMAND__A",
	0x00831ffe => "SCU_RAM_VERSION_HI__A",
	0x00831fff => "SCU_RAM_VERSION_LO__A",
	0x00c00000 => "ATV_COMM_EXEC__A",
	0x00c00001 => "ATV_COMM_STATE__A",
	0x00c00002 => "ATV_COMM_MB__A",
	0x00c00003 => "ATV_COMM_INT_REQ__A",
	0x00c00005 => "ATV_COMM_INT_STA__A",
	0x00c00006 => "ATV_COMM_INT_MSK__A",
	0x00c00007 => "ATV_COMM_INT_STM__A",
	0x00c0000f => "ATV_COMM_KEY__A",
	0x00c10000 => "ATV_TOP_COMM_EXEC__A",
	0x00c10001 => "ATV_TOP_COMM_STATE__A",
	0x00c10002 => "ATV_TOP_COMM_MB__A",
	0x00c10003 => "ATV_TOP_COMM_INT_REQ__A",
	0x00c10005 => "ATV_TOP_COMM_INT_STA__A",
	0x00c10006 => "ATV_TOP_COMM_INT_MSK__A",
	0x00c10007 => "ATV_TOP_COMM_INT_STM__A",
	0x00c1000f => "ATV_TOP_COMM_KEY__A",
	0x00c10010 => "ATV_TOP_CR_AMP_TH__A",
	0x00c10011 => "ATV_TOP_CR_CONT__A",
	0x00c10012 => "ATV_TOP_CR_OVM_TH__A",
	0x00c10013 => "ATV_TOP_NOISE_TH__A",
	0x00c10014 => "ATV_TOP_EQU0__A",
	0x00c10015 => "ATV_TOP_EQU1__A",
	0x00c10016 => "ATV_TOP_EQU2__A",
	0x00c10017 => "ATV_TOP_EQU3__A",
	0x00c10018 => "ATV_TOP_ROT_MODE__A",
	0x00c10019 => "ATV_TOP_MOD_CONTROL__A",
	0x00c1001a => "ATV_TOP_STD__A",
	0x00c1001b => "ATV_TOP_VID_AMP__A",
	0x00c1001c => "ATV_TOP_VID_PEAK__A",
	0x00c1001d => "ATV_TOP_FAGC_TH__A",
	0x00c1001e => "ATV_TOP_SYNC_SLICE__A",
	0x00c1001f => "ATV_TOP_SIF_GAIN__A",
	0x00c10020 => "ATV_TOP_SIF_TP__A",
	0x00c10021 => "ATV_TOP_MOD_ACCU__A",
	0x00c10022 => "ATV_TOP_CR_FREQ__A",
	0x00c10023 => "ATV_TOP_CR_PHAD__A",
	0x00c10024 => "ATV_TOP_AF_SIF_ATT__A",
	0x00c10025 => "ATV_TOP_STDBY__A",
	0x00c10026 => "ATV_TOP_OVERRIDE_SFR__A",
	0x00c10027 => "ATV_TOP_SFR_VID_GAIN__A",
	0x00c10028 => "ATV_TOP_SFR_AGC_RES__A",
	0x00c10029 => "ATV_TOP_OVM_COMP__A",
	0x00c1002a => "ATV_TOP_OUT_CONF__A",
	0x00c1002b => "ATV_TOP_CVBS_DBG_VALUE__A",
	0x00c1002c => "ATV_TOP_SIF_DBG_VALUE__A",
	0x00c1002d => "ATV_TOP_COMP_FREQ__A",
	0x00c1002e => "ATV_TOP_PHSHIFT__A",
	0x00c1002f => "ATV_TOP_COMP_AGC__A",
	0x00c10030 => "ATV_TOP_COMP_ACTIVE__A",
	0x00c10031 => "ATV_TOP_C2P_AMP__A",
	0x00ff0000 => "ATV_AFT_COMM_EXEC__A",
	0x00ff0010 => "ATV_AFT_TST__A",
	0x01000000 => "AUD_COMM_EXEC__A",
	0x01000002 => "AUD_COMM_MB__A",
	0x01010000 => "AUD_TOP_COMM_EXEC__A",
	0x01010002 => "AUD_TOP_COMM_MB__A",
	0x01010010 => "AUD_TOP_TR_MDE__A",
	0x01010011 => "AUD_TOP_TR_CTR__A",
	0x01010012 => "AUD_TOP_TR_RD_REG__A",
	0x01010013 => "AUD_TOP_TR_TIMER__A",
	0x01010014 => "AUD_TOP_DEMOD_TBO_SEL__A",
	0x01020023 => "AUD_DEM_RD_NIC_C_AD_BITS__A",
	0x01020038 => "AUD_DEM_RD_NIC_ADD_BITS_HI__A",
	0x01020038 => "AUD_DEM_RD_NIC_CIB__A",
	0x01020057 => "AUD_DEM_RD_NIC_ERROR_RATE__A",
	0x0102007e => "AUD_DEM_RD_STANDARD_RES__A",
	0x01020200 => "AUD_DEM_RD_STATUS__A",
	0x0102020f => "AUD_DEM_RD_RDS_ARRAY_CNT__A",
	0x01020210 => "AUD_DEM_RD_RDS_DATA__A",
	0x0103000f => "AUD_DEM_WR_FM_DEEMPH__A",
	0x01030017 => "AUD_DEM_WR_FM_DC_NOTCH_SW__A",
	0x01030020 => "AUD_DEM_WR_STANDARD_SEL__A",
	0x01030021 => "AUD_DEM_WR_NICAM_THRSHLD__A",
	0x01030022 => "AUD_DEM_WR_A2_THRSHLD__A",
	0x01030023 => "AUD_DEM_WR_BTSC_THRSHLD__A",
	0x01030024 => "AUD_DEM_WR_CM_A_THRSHLD__A",
	0x01030025 => "AUD_DEM_WR_CM_B_THRSHLD__A",
	0x01030030 => "AUD_DEM_WR_MODUS__A",
	0x01030050 => "AUD_DEM_WR_I2S_CONFIG2__A",
	0x0103006f => "AUD_DEM_WR_FM_MATRIX__A",
	0x01030093 => "AUD_DEM_WR_DCO_B_LO__A",
	0x0103009b => "AUD_DEM_WR_DCO_B_HI__A",
	0x010300a3 => "AUD_DEM_WR_DCO_A_LO__A",
	0x010300ab => "AUD_DEM_WR_DCO_A_HI__A",
	0x01040018 => "AUD_DSP_RD_FM_IDENT_VALUE__A",
	0x01040019 => "AUD_DSP_RD_QPEAK_L__A",
	0x0104001a => "AUD_DSP_RD_QPEAK_R__A",
	0x0104001b => "AUD_DSP_RD_FM_DC_LEVEL_A__A",
	0x0104001c => "AUD_DSP_RD_FM_DC_LEVEL_B__A",
	0x0104001d => "AUD_DSP_RD_XDFP_FW__A",
	0x0104007b => "AUD_DSP_RD_STATUS2__A",
	0x010404b8 => "AUD_DSP_RD_XFP_FW__A",
	0x01050000 => "AUD_DSP_WR_VOLUME__A",
	0x0105000c => "AUD_DSP_WR_QPEAK__A",
	0x0105000e => "AUD_DSP_WR_FM_PRESC__A",
	0x01050010 => "AUD_DSP_WR_NICAM_PRESC__A",
	0x01050014 => "AUD_DSP_WR_BEEPER__A",
	0x01050026 => "AUD_DSP_WR_SYNC_OUT__A",
	0x01050029 => "AUD_DSP_WR_AVC__A",
	0x0105002a => "AUD_DSP_WR_I2S_OUT_FS__A",
	0x0105002b => "AUD_DSP_WR_AV_SYNC__A",
	0x01050038 => "AUD_DSP_WR_SRC_I2S_MATR__A",
	0x01060000 => "AUD_XFP_DRAM_1K__A",
	0x01070000 => "AUD_XFP_PRAM_4K__A",
	0x01080000 => "AUD_XDFP_DRAM_1K__A",
	0x01090000 => "AUD_XDFP_PRAM_4K__A",
	0x01400000 => "QAM_COMM_EXEC__A",
	0x01400002 => "QAM_COMM_MB__A",
	0x01400003 => "QAM_COMM_INT_REQ__A",
	0x01400005 => "QAM_COMM_INT_STA__A",
	0x01400006 => "QAM_COMM_INT_MSK__A",
	0x01400007 => "QAM_COMM_INT_STM__A",
	0x01410000 => "QAM_TOP_COMM_EXEC__A",
	0x01410010 => "QAM_TOP_ANNEX__A",
	0x01410011 => "QAM_TOP_CONSTELLATION__A",
	0x01420000 => "QAM_FQ_COMM_EXEC__A",
	0x01420010 => "QAM_FQ_MODE__A",
	0x01420011 => "QAM_FQ_MU_FACTOR__A",
	0x01420012 => "QAM_FQ_LA_FACTOR__A",
	0x01420016 => "QAM_FQ_CENTTAP_IDX__A",
	0x01420017 => "QAM_FQ_CENTTAP_VALUE__A",
	0x01420020 => "QAM_FQ_TAP_RE_EL0__A",
	0x01420021 => "QAM_FQ_TAP_IM_EL0__A",
	0x01420022 => "QAM_FQ_TAP_RE_EL1__A",
	0x01420023 => "QAM_FQ_TAP_IM_EL1__A",
	0x01420024 => "QAM_FQ_TAP_RE_EL2__A",
	0x01420025 => "QAM_FQ_TAP_IM_EL2__A",
	0x01420026 => "QAM_FQ_TAP_RE_EL3__A",
	0x01420027 => "QAM_FQ_TAP_IM_EL3__A",
	0x01420028 => "QAM_FQ_TAP_RE_EL4__A",
	0x01420029 => "QAM_FQ_TAP_IM_EL4__A",
	0x0142002a => "QAM_FQ_TAP_RE_EL5__A",
	0x0142002b => "QAM_FQ_TAP_IM_EL5__A",
	0x0142002c => "QAM_FQ_TAP_RE_EL6__A",
	0x0142002d => "QAM_FQ_TAP_IM_EL6__A",
	0x0142002e => "QAM_FQ_TAP_RE_EL7__A",
	0x0142002f => "QAM_FQ_TAP_IM_EL7__A",
	0x01420030 => "QAM_FQ_TAP_RE_EL8__A",
	0x01420031 => "QAM_FQ_TAP_IM_EL8__A",
	0x01420032 => "QAM_FQ_TAP_RE_EL9__A",
	0x01420033 => "QAM_FQ_TAP_IM_EL9__A",
	0x01420034 => "QAM_FQ_TAP_RE_EL10__A",
	0x01420035 => "QAM_FQ_TAP_IM_EL10__A",
	0x01420036 => "QAM_FQ_TAP_RE_EL11__A",
	0x01420037 => "QAM_FQ_TAP_IM_EL11__A",
	0x01420038 => "QAM_FQ_TAP_RE_EL12__A",
	0x01420039 => "QAM_FQ_TAP_IM_EL12__A",
	0x0142003a => "QAM_FQ_TAP_RE_EL13__A",
	0x0142003b => "QAM_FQ_TAP_IM_EL13__A",
	0x0142003c => "QAM_FQ_TAP_RE_EL14__A",
	0x0142003d => "QAM_FQ_TAP_IM_EL14__A",
	0x0142003e => "QAM_FQ_TAP_RE_EL15__A",
	0x0142003f => "QAM_FQ_TAP_IM_EL15__A",
	0x01420040 => "QAM_FQ_TAP_RE_EL16__A",
	0x01420041 => "QAM_FQ_TAP_IM_EL16__A",
	0x01420042 => "QAM_FQ_TAP_RE_EL17__A",
	0x01420043 => "QAM_FQ_TAP_IM_EL17__A",
	0x01420044 => "QAM_FQ_TAP_RE_EL18__A",
	0x01420045 => "QAM_FQ_TAP_IM_EL18__A",
	0x01420046 => "QAM_FQ_TAP_RE_EL19__A",
	0x01420047 => "QAM_FQ_TAP_IM_EL19__A",
	0x01420048 => "QAM_FQ_TAP_RE_EL20__A",
	0x01420049 => "QAM_FQ_TAP_IM_EL20__A",
	0x0142004a => "QAM_FQ_TAP_RE_EL21__A",
	0x0142004b => "QAM_FQ_TAP_IM_EL21__A",
	0x0142004c => "QAM_FQ_TAP_RE_EL22__A",
	0x0142004d => "QAM_FQ_TAP_IM_EL22__A",
	0x0142004e => "QAM_FQ_TAP_RE_EL23__A",
	0x0142004f => "QAM_FQ_TAP_IM_EL23__A",
	0x01430000 => "QAM_SL_COMM_EXEC__A",
	0x01430002 => "QAM_SL_COMM_MB__A",
	0x01430003 => "QAM_SL_COMM_INT_REQ__A",
	0x01430005 => "QAM_SL_COMM_INT_STA__A",
	0x01430006 => "QAM_SL_COMM_INT_MSK__A",
	0x01430007 => "QAM_SL_COMM_INT_STM__A",
	0x01430010 => "QAM_SL_MODE__A",
	0x01430011 => "QAM_SL_K_FACTOR__A",
	0x01430012 => "QAM_SL_MEDIAN__A",
	0x01430013 => "QAM_SL_ALPHA__A",
	0x01430014 => "QAM_SL_PHASELIMIT__A",
	0x01430015 => "QAM_SL_MTA_LENGTH__A",
	0x01430016 => "QAM_SL_MEDIAN_ERROR__A",
	0x01430017 => "QAM_SL_ERR_POWER__A",
	0x01430018 => "QAM_SL_QUAL_QAM_4_0__A",
	0x01430019 => "QAM_SL_QUAL_QAM_8_0__A",
	0x0143001a => "QAM_SL_QUAL_QAM_16_0__A",
	0x0143001b => "QAM_SL_QUAL_QAM_16_1__A",
	0x0143001c => "QAM_SL_QUAL_QAM_32_0__A",
	0x0143001d => "QAM_SL_QUAL_QAM_32_1__A",
	0x0143001e => "QAM_SL_QUAL_QAM_32_2__A",
	0x0143001f => "QAM_SL_QUAL_QAM_64_0__A",
	0x01430020 => "QAM_SL_QUAL_QAM_64_1__A",
	0x01430021 => "QAM_SL_QUAL_QAM_64_2__A",
	0x01430022 => "QAM_SL_QUAL_QAM_64_3__A",
	0x01430023 => "QAM_SL_QUAL_QAM_128_0__A",
	0x01430024 => "QAM_SL_QUAL_QAM_128_1__A",
	0x01430025 => "QAM_SL_QUAL_QAM_128_2__A",
	0x01430026 => "QAM_SL_QUAL_QAM_128_3__A",
	0x01430027 => "QAM_SL_QUAL_QAM_128_4__A",
	0x01430028 => "QAM_SL_QUAL_QAM_128_5__A",
	0x01430029 => "QAM_SL_QUAL_QAM_128_5H__A",
	0x0143002a => "QAM_SL_QUAL_QAM_256_0__A",
	0x0143002b => "QAM_SL_QUAL_QAM_256_1__A",
	0x0143002c => "QAM_SL_QUAL_QAM_256_2__A",
	0x0143002d => "QAM_SL_QUAL_QAM_256_3__A",
	0x0143002e => "QAM_SL_QUAL_QAM_256_4__A",
	0x0143002f => "QAM_SL_QUAL_QAM_256_5__A",
	0x01430030 => "QAM_SL_QUAL_QAM_256_5H__A",
	0x01430031 => "QAM_SL_QUAL_QAM_256_6__A",
	0x01430032 => "QAM_SL_QUAL_QAM_256_6H__A",
	0x01430033 => "QAM_SL_QUAL_QAM_256_7__A",
	0x01430034 => "QAM_SL_QUAL_QAM_256_7H__A",
	0x01440000 => "QAM_DQ_COMM_EXEC__A",
	0x01440010 => "QAM_DQ_MODE__A",
	0x01440011 => "QAM_DQ_MU_FACTOR__A",
	0x01440012 => "QAM_DQ_LA_FACTOR__A",
	0x01440013 => "QAM_DQ_CMA_RATIO__A",
	0x01440014 => "QAM_DQ_QUAL_RADSEL__A",
	0x01440015 => "QAM_DQ_QUAL_ENA__A",
	0x01440018 => "QAM_DQ_QUAL_FUN0__A",
	0x01440019 => "QAM_DQ_QUAL_FUN1__A",
	0x0144001a => "QAM_DQ_QUAL_FUN2__A",
	0x0144001b => "QAM_DQ_QUAL_FUN3__A",
	0x0144001c => "QAM_DQ_QUAL_FUN4__A",
	0x0144001d => "QAM_DQ_QUAL_FUN5__A",
	0x0144001e => "QAM_DQ_RAW_LIM__A",
	0x01440020 => "QAM_DQ_TAP_RE_EL0__A",
	0x01440021 => "QAM_DQ_TAP_IM_EL0__A",
	0x01440022 => "QAM_DQ_TAP_RE_EL1__A",
	0x01440023 => "QAM_DQ_TAP_IM_EL1__A",
	0x01440024 => "QAM_DQ_TAP_RE_EL2__A",
	0x01440025 => "QAM_DQ_TAP_IM_EL2__A",
	0x01440026 => "QAM_DQ_TAP_RE_EL3__A",
	0x01440027 => "QAM_DQ_TAP_IM_EL3__A",
	0x01440028 => "QAM_DQ_TAP_RE_EL4__A",
	0x01440029 => "QAM_DQ_TAP_IM_EL4__A",
	0x0144002a => "QAM_DQ_TAP_RE_EL5__A",
	0x0144002b => "QAM_DQ_TAP_IM_EL5__A",
	0x0144002c => "QAM_DQ_TAP_RE_EL6__A",
	0x0144002d => "QAM_DQ_TAP_IM_EL6__A",
	0x0144002e => "QAM_DQ_TAP_RE_EL7__A",
	0x0144002f => "QAM_DQ_TAP_IM_EL7__A",
	0x01440030 => "QAM_DQ_TAP_RE_EL8__A",
	0x01440031 => "QAM_DQ_TAP_IM_EL8__A",
	0x01440032 => "QAM_DQ_TAP_RE_EL9__A",
	0x01440033 => "QAM_DQ_TAP_IM_EL9__A",
	0x01440034 => "QAM_DQ_TAP_RE_EL10__A",
	0x01440035 => "QAM_DQ_TAP_IM_EL10__A",
	0x01440036 => "QAM_DQ_TAP_RE_EL11__A",
	0x01440037 => "QAM_DQ_TAP_IM_EL11__A",
	0x01440038 => "QAM_DQ_TAP_RE_EL12__A",
	0x01440039 => "QAM_DQ_TAP_IM_EL12__A",
	0x0144003a => "QAM_DQ_TAP_RE_EL13__A",
	0x0144003b => "QAM_DQ_TAP_IM_EL13__A",
	0x0144003c => "QAM_DQ_TAP_RE_EL14__A",
	0x0144003d => "QAM_DQ_TAP_IM_EL14__A",
	0x0144003e => "QAM_DQ_TAP_RE_EL15__A",
	0x0144003f => "QAM_DQ_TAP_IM_EL15__A",
	0x01440040 => "QAM_DQ_TAP_RE_EL16__A",
	0x01440041 => "QAM_DQ_TAP_IM_EL16__A",
	0x01440042 => "QAM_DQ_TAP_RE_EL17__A",
	0x01440043 => "QAM_DQ_TAP_IM_EL17__A",
	0x01440044 => "QAM_DQ_TAP_RE_EL18__A",
	0x01440045 => "QAM_DQ_TAP_IM_EL18__A",
	0x01440046 => "QAM_DQ_TAP_RE_EL19__A",
	0x01440047 => "QAM_DQ_TAP_IM_EL19__A",
	0x01440048 => "QAM_DQ_TAP_RE_EL20__A",
	0x01440049 => "QAM_DQ_TAP_IM_EL20__A",
	0x0144004a => "QAM_DQ_TAP_RE_EL21__A",
	0x0144004b => "QAM_DQ_TAP_IM_EL21__A",
	0x0144004c => "QAM_DQ_TAP_RE_EL22__A",
	0x0144004d => "QAM_DQ_TAP_IM_EL22__A",
	0x0144004e => "QAM_DQ_TAP_RE_EL23__A",
	0x0144004f => "QAM_DQ_TAP_IM_EL23__A",
	0x01440050 => "QAM_DQ_TAP_RE_EL24__A",
	0x01440051 => "QAM_DQ_TAP_IM_EL24__A",
	0x01440052 => "QAM_DQ_TAP_RE_EL25__A",
	0x01440053 => "QAM_DQ_TAP_IM_EL25__A",
	0x01440054 => "QAM_DQ_TAP_RE_EL26__A",
	0x01440055 => "QAM_DQ_TAP_IM_EL26__A",
	0x01440056 => "QAM_DQ_TAP_RE_EL27__A",
	0x01440057 => "QAM_DQ_TAP_IM_EL27__A",
	0x01450000 => "QAM_LC_COMM_EXEC__A",
	0x01450002 => "QAM_LC_COMM_MB__A",
	0x01450003 => "QAM_LC_COMM_INT_REQ__A",
	0x01450005 => "QAM_LC_COMM_INT_STA__A",
	0x01450006 => "QAM_LC_COMM_INT_MSK__A",
	0x01450007 => "QAM_LC_COMM_INT_STM__A",
	0x01450010 => "QAM_LC_MODE__A",
	0x01450011 => "QAM_LC_CA__A",
	0x01450012 => "QAM_LC_CF__A",
	0x01450013 => "QAM_LC_CF1__A",
	0x01450014 => "QAM_LC_CP__A",
	0x01450015 => "QAM_LC_CI__A",
	0x01450016 => "QAM_LC_EP__A",
	0x01450017 => "QAM_LC_EI__A",
	0x01450018 => "QAM_LC_QUAL_TAB0__A",
	0x01450019 => "QAM_LC_QUAL_TAB1__A",
	0x0145001a => "QAM_LC_QUAL_TAB2__A",
	0x0145001b => "QAM_LC_QUAL_TAB3__A",
	0x0145001c => "QAM_LC_QUAL_TAB4__A",
	0x0145001d => "QAM_LC_QUAL_TAB5__A",
	0x0145001e => "QAM_LC_QUAL_TAB6__A",
	0x0145001f => "QAM_LC_QUAL_TAB8__A",
	0x01450020 => "QAM_LC_QUAL_TAB9__A",
	0x01450021 => "QAM_LC_QUAL_TAB10__A",
	0x01450022 => "QAM_LC_QUAL_TAB12__A",
	0x01450023 => "QAM_LC_QUAL_TAB15__A",
	0x01450024 => "QAM_LC_QUAL_TAB16__A",
	0x01450025 => "QAM_LC_QUAL_TAB20__A",
	0x01450026 => "QAM_LC_QUAL_TAB25__A",
	0x01450027 => "QAM_LC_EQ_TIMING__A",
	0x01450028 => "QAM_LC_LPF_FACTORP__A",
	0x01450029 => "QAM_LC_LPF_FACTORI__A",
	0x0145002a => "QAM_LC_RATE_LIMIT__A",
	0x0145002b => "QAM_LC_SYMBOL_FREQ__A",
	0x0145002c => "QAM_LC_MTA_LENGTH__A",
	0x0145002d => "QAM_LC_AMP_ACCU__A",
	0x0145002e => "QAM_LC_FREQ_ACCU__A",
	0x0145002f => "QAM_LC_RATE_ACCU__A",
	0x01450030 => "QAM_LC_AMPLITUDE__A",
	0x01450031 => "QAM_LC_RAD_ERROR__A",
	0x01450032 => "QAM_LC_FREQ_OFFS__A",
	0x01450033 => "QAM_LC_PHASE_ERROR__A",
	0x01470000 => "QAM_SY_COMM_EXEC__A",
	0x01470002 => "QAM_SY_COMM_MB__A",
	0x01470003 => "QAM_SY_COMM_INT_REQ__A",
	0x01470005 => "QAM_SY_COMM_INT_STA__A",
	0x01470006 => "QAM_SY_COMM_INT_MSK__A",
	0x01470007 => "QAM_SY_COMM_INT_STM__A",
	0x01470010 => "QAM_SY_STATUS__A",
	0x01470011 => "QAM_SY_TIMEOUT__A",
	0x01470012 => "QAM_SY_SYNC_LWM__A",
	0x01470013 => "QAM_SY_SYNC_AWM__A",
	0x01470014 => "QAM_SY_SYNC_HWM__A",
	0x01470015 => "QAM_SY_UNLOCK__A",
	0x01470016 => "QAM_SY_CONTROL_WORD__A",
	0x01470017 => "QAM_SY_SP_INV__A",
	0x01480000 => "QAM_VD_ISS_RAM__A",
	0x01490000 => "QAM_VD_QSS_RAM__A",
	0x014a0000 => "QAM_VD_SYM_RAM__A",
	0x01800000 => "IQM_COMM_EXEC__A",
	0x01800002 => "IQM_COMM_MB__A",
	0x01800003 => "IQM_COMM_INT_REQ__A",
	0x01800005 => "IQM_COMM_INT_STA__A",
	0x01800006 => "IQM_COMM_INT_MSK__A",
	0x01800007 => "IQM_COMM_INT_STM__A",
	0x01820000 => "IQM_FS_COMM_EXEC__A",
	0x01820002 => "IQM_FS_COMM_MB__A",
	0x01820010 => "IQM_FS_RATE_OFS_LO__A",
	0x01820011 => "IQM_FS_RATE_OFS_HI__A",
	0x01820012 => "IQM_FS_RATE_LO__A",
	0x01820013 => "IQM_FS_RATE_HI__A",
	0x01820014 => "IQM_FS_ADJ_SEL__A",
	0x01830000 => "IQM_FD_COMM_EXEC__A",
	0x01830002 => "IQM_FD_COMM_MB__A",
	0x01830010 => "IQM_FD_RATESEL__A",
	0x01840000 => "IQM_RC_COMM_EXEC__A",
	0x01840002 => "IQM_RC_COMM_MB__A",
	0x01840010 => "IQM_RC_RATE_OFS_LO__A",
	0x01840011 => "IQM_RC_RATE_OFS_HI__A",
	0x01840012 => "IQM_RC_RATE_LO__A",
	0x01840013 => "IQM_RC_RATE_HI__A",
	0x01840014 => "IQM_RC_ADJ_SEL__A",
	0x01840015 => "IQM_RC_CROUT_ENA__A",
	0x01840016 => "IQM_RC_STRETCH__A",
	0x01850000 => "IQM_RT_COMM_EXEC__A",
	0x01850002 => "IQM_RT_COMM_MB__A",
	0x01850010 => "IQM_RT_ACTIVE__A",
	0x01850011 => "IQM_RT_LO_INCR__A",
	0x01850012 => "IQM_RT_ROT_BP__A",
	0x01850013 => "IQM_RT_LP_BP__A",
	0x01850014 => "IQM_RT_DELAY__A",
	0x01860000 => "IQM_CF_COMM_EXEC__A",
	0x01860002 => "IQM_CF_COMM_MB__A",
	0x01860003 => "IQM_CF_COMM_INT_REQ__A",
	0x01860005 => "IQM_CF_COMM_INT_STA__A",
	0x01860006 => "IQM_CF_COMM_INT_MSK__A",
	0x01860007 => "IQM_CF_COMM_INT_STM__A",
	0x01860010 => "IQM_CF_SYMMETRIC__A",
	0x01860011 => "IQM_CF_MIDTAP__A",
	0x01860012 => "IQM_CF_OUT_ENA__A",
	0x01860013 => "IQM_CF_ADJ_SEL__A",
	0x01860014 => "IQM_CF_SCALE__A",
	0x01860015 => "IQM_CF_SCALE_SH__A",
	0x01860016 => "IQM_CF_AMP__A",
	0x01860017 => "IQM_CF_POW_MEAS_LEN__A",
	0x01860018 => "IQM_CF_POW__A",
	0x01860019 => "IQM_CF_DS_ENA__A",
	0x0186001a => "IQM_CF_POW_UPD__A",
	0x01860020 => "IQM_CF_TAP_RE0__A",
	0x01860021 => "IQM_CF_TAP_RE1__A",
	0x01860022 => "IQM_CF_TAP_RE2__A",
	0x01860023 => "IQM_CF_TAP_RE3__A",
	0x01860024 => "IQM_CF_TAP_RE4__A",
	0x01860025 => "IQM_CF_TAP_RE5__A",
	0x01860026 => "IQM_CF_TAP_RE6__A",
	0x01860027 => "IQM_CF_TAP_RE7__A",
	0x01860028 => "IQM_CF_TAP_RE8__A",
	0x01860029 => "IQM_CF_TAP_RE9__A",
	0x0186002a => "IQM_CF_TAP_RE10__A",
	0x0186002b => "IQM_CF_TAP_RE11__A",
	0x0186002c => "IQM_CF_TAP_RE12__A",
	0x0186002d => "IQM_CF_TAP_RE13__A",
	0x0186002e => "IQM_CF_TAP_RE14__A",
	0x0186002f => "IQM_CF_TAP_RE15__A",
	0x01860030 => "IQM_CF_TAP_RE16__A",
	0x01860031 => "IQM_CF_TAP_RE17__A",
	0x01860032 => "IQM_CF_TAP_RE18__A",
	0x01860033 => "IQM_CF_TAP_RE19__A",
	0x01860034 => "IQM_CF_TAP_RE20__A",
	0x01860035 => "IQM_CF_TAP_RE21__A",
	0x01860036 => "IQM_CF_TAP_RE22__A",
	0x01860037 => "IQM_CF_TAP_RE23__A",
	0x01860038 => "IQM_CF_TAP_RE24__A",
	0x01860039 => "IQM_CF_TAP_RE25__A",
	0x0186003a => "IQM_CF_TAP_RE26__A",
	0x0186003b => "IQM_CF_TAP_RE27__A",
	0x01860040 => "IQM_CF_TAP_IM0__A",
	0x01860041 => "IQM_CF_TAP_IM1__A",
	0x01860042 => "IQM_CF_TAP_IM2__A",
	0x01860043 => "IQM_CF_TAP_IM3__A",
	0x01860044 => "IQM_CF_TAP_IM4__A",
	0x01860045 => "IQM_CF_TAP_IM5__A",
	0x01860046 => "IQM_CF_TAP_IM6__A",
	0x01860047 => "IQM_CF_TAP_IM7__A",
	0x01860048 => "IQM_CF_TAP_IM8__A",
	0x01860049 => "IQM_CF_TAP_IM9__A",
	0x0186004a => "IQM_CF_TAP_IM10__A",
	0x0186004b => "IQM_CF_TAP_IM11__A",
	0x0186004c => "IQM_CF_TAP_IM12__A",
	0x0186004d => "IQM_CF_TAP_IM13__A",
	0x0186004e => "IQM_CF_TAP_IM14__A",
	0x0186004f => "IQM_CF_TAP_IM15__A",
	0x01860050 => "IQM_CF_TAP_IM16__A",
	0x01860051 => "IQM_CF_TAP_IM17__A",
	0x01860052 => "IQM_CF_TAP_IM18__A",
	0x01860053 => "IQM_CF_TAP_IM19__A",
	0x01860054 => "IQM_CF_TAP_IM20__A",
	0x01860055 => "IQM_CF_TAP_IM21__A",
	0x01860056 => "IQM_CF_TAP_IM22__A",
	0x01860057 => "IQM_CF_TAP_IM23__A",
	0x01860058 => "IQM_CF_TAP_IM24__A",
	0x01860059 => "IQM_CF_TAP_IM25__A",
	0x0186005a => "IQM_CF_TAP_IM26__A",
	0x0186005b => "IQM_CF_TAP_IM27__A",
	0x01860060 => "IQM_CF_CLP_VAL__A",
	0x01860061 => "IQM_CF_DATATH__A",
	0x01860062 => "IQM_CF_PKDTH__A",
	0x01860063 => "IQM_CF_WND_LEN__A",
	0x01860064 => "IQM_CF_DET_LCT__A",
	0x01860065 => "IQM_CF_SNS_LEN__A",
	0x01860066 => "IQM_CF_SNS_SENSE__A",
	0x01860067 => "IQM_CF_BYPASSDET__A",
	0x01860068 => "IQM_CF_UPD_ENA__A",
	0x01870000 => "IQM_AF_COMM_EXEC__A",
	0x01870002 => "IQM_AF_COMM_MB__A",
	0x01870003 => "IQM_AF_COMM_INT_REQ__A",
	0x01870005 => "IQM_AF_COMM_INT_STA__A",
	0x01870006 => "IQM_AF_COMM_INT_MSK__A",
	0x01870007 => "IQM_AF_COMM_INT_STM__A",
	0x01870010 => "IQM_AF_FDB_SEL__A",
	0x01870012 => "IQM_AF_CLKNEG__A",
	0x01870013 => "IQM_AF_MON_IN_MUX__A",
	0x01870014 => "IQM_AF_MON_IN5__A",
	0x01870015 => "IQM_AF_MON_IN4__A",
	0x01870016 => "IQM_AF_MON_IN3__A",
	0x01870017 => "IQM_AF_MON_IN2__A",
	0x01870018 => "IQM_AF_MON_IN1__A",
	0x01870019 => "IQM_AF_MON_IN0__A",
	0x0187001a => "IQM_AF_MON_IN_VAL__A",
	0x0187001b => "IQM_AF_START_LOCK__A",
	0x0187001c => "IQM_AF_PHASE0__A",
	0x0187001d => "IQM_AF_PHASE1__A",
	0x0187001e => "IQM_AF_PHASE2__A",
	0x0187001f => "IQM_AF_SCU_PHASE__A",
	0x01870020 => "IQM_AF_SYNC_SEL__A",
	0x01870021 => "IQM_AF_ADC_CONF__A",
	0x01870022 => "IQM_AF_CLP_CLIP__A",
	0x01870023 => "IQM_AF_CLP_LEN__A",
	0x01870024 => "IQM_AF_CLP_TH__A",
	0x01870025 => "IQM_AF_DCF_BYPASS__A",
	0x01870026 => "IQM_AF_SNS_LEN__A",
	0x01870027 => "IQM_AF_SNS_SENSE__A",
	0x01870028 => "IQM_AF_AGC_IF__A",
	0x01870029 => "IQM_AF_AGC_RF__A",
	0x0187002b => "IQM_AF_PDREF__A",
	0x0187002c => "IQM_AF_STDBY__A",
	0x0187002d => "IQM_AF_AMUX__A",
	0x0187002e => "IQM_AF_TST_AFEMAIN__A",
	0x0187002f => "IQM_AF_UPD_SEL__A",
	0x01870030 => "IQM_AF_INC_DATATH__A",
	0x01870031 => "IQM_AF_INC_PKDTH__A",
	0x01870032 => "IQM_AF_INC_WND_LEN__A",
	0x01870033 => "IQM_AF_INC_DLY__A",
	0x01870034 => "IQM_AF_INC_LCT__A",
	0x01870035 => "IQM_AF_INC_CLP_VAL__A",
	0x01870036 => "IQM_AF_INC_BYPASS__A",
	0x01870037 => "IQM_AF_INC_MODE_SEL__A",
	0x01870038 => "IQM_AF_INC_A_DLY__A",
	0x01870039 => "IQM_AF_ISNS_LEN__A",
	0x0187003a => "IQM_AF_ISNS_SENSE__A",
	0x0187003b => "IQM_AF_CMP_STATE__A",
	0x0187003c => "IQM_AF_CMP_DC_OUT__A",
	0x0187003d => "IQM_AF_CMP_DC_IN__A",
	0x0187003e => "IQM_AF_CMP_AMP__A",
	0x0187003f => "IQM_AF_CMP_DN_AVG__A",
	0x01870040 => "IQM_AF_CMP_ACTIVE__A",
	0x01870080 => "IQM_AF_CMP_MEM0__A",
	0x01870081 => "IQM_AF_CMP_MEM1__A",
	0x01870082 => "IQM_AF_CMP_MEM2__A",
	0x01870083 => "IQM_AF_CMP_MEM3__A",
	0x01870084 => "IQM_AF_CMP_MEM4__A",
	0x01870085 => "IQM_AF_CMP_MEM5__A",
	0x01870086 => "IQM_AF_CMP_MEM6__A",
	0x01870087 => "IQM_AF_CMP_MEM7__A",
	0x01870088 => "IQM_AF_CMP_MEM8__A",
	0x01870089 => "IQM_AF_CMP_MEM9__A",
	0x0187008a => "IQM_AF_CMP_MEM10__A",
	0x0187008b => "IQM_AF_CMP_MEM11__A",
	0x0187008c => "IQM_AF_CMP_MEM12__A",
	0x0187008d => "IQM_AF_CMP_MEM13__A",
	0x0187008e => "IQM_AF_CMP_MEM14__A",
	0x0187008f => "IQM_AF_CMP_MEM15__A",
	0x01870090 => "IQM_AF_CMP_MEM16__A",
	0x01870091 => "IQM_AF_CMP_MEM17__A",
	0x01870092 => "IQM_AF_CMP_MEM18__A",
	0x01870093 => "IQM_AF_CMP_MEM19__A",
	0x01870094 => "IQM_AF_CMP_MEM20__A",
	0x01870095 => "IQM_AF_CMP_MEM21__A",
	0x01870096 => "IQM_AF_CMP_MEM22__A",
	0x01870097 => "IQM_AF_CMP_MEM23__A",
	0x01870098 => "IQM_AF_CMP_MEM24__A",
	0x01870099 => "IQM_AF_CMP_MEM25__A",
	0x0187009a => "IQM_AF_CMP_MEM26__A",
	0x0187009b => "IQM_AF_CMP_MEM27__A",
	0x0187009c => "IQM_AF_CMP_MEM28__A",
	0x0187009d => "IQM_AF_CMP_MEM29__A",
	0x0187009e => "IQM_AF_CMP_MEM30__A",
	0x0187009f => "IQM_AF_CMP_MEM31__A",
	0x018700a0 => "IQM_AF_CMP_MEM32__A",
	0x018700a1 => "IQM_AF_CMP_MEM33__A",
	0x018700a2 => "IQM_AF_CMP_MEM34__A",
	0x018700a3 => "IQM_AF_CMP_MEM35__A",
	0x018700a4 => "IQM_AF_CMP_MEM36__A",
	0x018700a5 => "IQM_AF_CMP_MEM37__A",
	0x018700a6 => "IQM_AF_CMP_MEM38__A",
	0x018700a7 => "IQM_AF_CMP_MEM39__A",
	0x018700a8 => "IQM_AF_CMP_MEM40__A",
	0x018700a9 => "IQM_AF_CMP_MEM41__A",
	0x018700aa => "IQM_AF_CMP_MEM42__A",
	0x018700ab => "IQM_AF_CMP_MEM43__A",
	0x018700ac => "IQM_AF_CMP_MEM44__A",
	0x018700ad => "IQM_AF_CMP_MEM45__A",
	0x018700ae => "IQM_AF_CMP_MEM46__A",
	0x018700af => "IQM_AF_CMP_MEM47__A",
	0x018700b0 => "IQM_AF_CMP_MEM48__A",
	0x018700b1 => "IQM_AF_CMP_MEM49__A",
	0x018700b2 => "IQM_AF_CMP_MEM50__A",
	0x018700b3 => "IQM_AF_CMP_MEM51__A",
	0x018700b4 => "IQM_AF_CMP_MEM52__A",
	0x018700b5 => "IQM_AF_CMP_MEM53__A",
	0x018700b6 => "IQM_AF_CMP_MEM54__A",
	0x018700b7 => "IQM_AF_CMP_MEM55__A",
	0x018700b8 => "IQM_AF_CMP_MEM56__A",
	0x018700b9 => "IQM_AF_CMP_MEM57__A",
	0x018700ba => "IQM_AF_CMP_MEM58__A",
	0x018700bb => "IQM_AF_CMP_MEM59__A",
	0x018700bc => "IQM_AF_CMP_MEM60__A",
	0x018700bd => "IQM_AF_CMP_MEM61__A",
	0x018700be => "IQM_AF_CMP_MEM62__A",
	0x018700bf => "IQM_AF_CMP_MEM63__A",
	0x01880000 => "IQM_RT_RAM__A",
	0x01c00000 => "FEC_COMM_EXEC__A",
	0x01c00002 => "FEC_COMM_MB__A",
	0x01c00003 => "FEC_COMM_INT_REQ__A",
	0x01c00005 => "FEC_COMM_INT_STA__A",
	0x01c00006 => "FEC_COMM_INT_MSK__A",
	0x01c00007 => "FEC_COMM_INT_STM__A",
	0x01c10000 => "FEC_TOP_COMM_EXEC__A",
	0x01c10010 => "FEC_TOP_ANNEX__A",
	0x01c20000 => "FEC_DI_COMM_EXEC__A",
	0x01c20002 => "FEC_DI_COMM_MB__A",
	0x01c20003 => "FEC_DI_COMM_INT_REQ__A",
	0x01c20005 => "FEC_DI_COMM_INT_STA__A",
	0x01c20006 => "FEC_DI_COMM_INT_MSK__A",
	0x01c20007 => "FEC_DI_COMM_INT_STM__A",
	0x01c20010 => "FEC_DI_STATUS__A",
	0x01c20011 => "FEC_DI_MODE__A",
	0x01c20012 => "FEC_DI_CONTROL_WORD__A",
	0x01c20013 => "FEC_DI_RESTART__A",
	0x01c20014 => "FEC_DI_TIMEOUT_LO__A",
	0x01c20015 => "FEC_DI_TIMEOUT_HI__A",
	0x01c20016 => "FEC_DI_INPUT_CTL__A",
	0x01c30000 => "FEC_RS_COMM_EXEC__A",
	0x01c30002 => "FEC_RS_COMM_MB__A",
	0x01c30003 => "FEC_RS_COMM_INT_REQ__A",
	0x01c30005 => "FEC_RS_COMM_INT_STA__A",
	0x01c30006 => "FEC_RS_COMM_INT_MSK__A",
	0x01c30007 => "FEC_RS_COMM_INT_STM__A",
	0x01c30010 => "FEC_RS_STATUS__A",
	0x01c30011 => "FEC_RS_MODE__A",
	0x01c30012 => "FEC_RS_MEASUREMENT_PERIOD__A",
	0x01c30013 => "FEC_RS_MEASUREMENT_PRESCALE__A",
	0x01c30014 => "FEC_RS_NR_BIT_ERRORS__A",
	0x01c30015 => "FEC_RS_NR_SYMBOL_ERRORS__A",
	0x01c30016 => "FEC_RS_NR_PACKET_ERRORS__A",
	0x01c30017 => "FEC_RS_NR_FAILURES__A",
	0x01c40000 => "FEC_OC_COMM_EXEC__A",
	0x01c40002 => "FEC_OC_COMM_MB__A",
	0x01c40003 => "FEC_OC_COMM_INT_REQ__A",
	0x01c40005 => "FEC_OC_COMM_INT_STA__A",
	0x01c40006 => "FEC_OC_COMM_INT_MSK__A",
	0x01c40007 => "FEC_OC_COMM_INT_STM__A",
	0x01c40010 => "FEC_OC_STATUS__A",
	0x01c40011 => "FEC_OC_MODE__A",
	0x01c40012 => "FEC_OC_DPR_MODE__A",
	0x01c40013 => "FEC_OC_DPR_UNLOCK__A",
	0x01c40014 => "FEC_OC_DTO_MODE__A",
	0x01c40015 => "FEC_OC_DTO_PERIOD__A",
	0x01c40016 => "FEC_OC_DTO_RATE_LO__A",
	0x01c40017 => "FEC_OC_DTO_RATE_HI__A",
	0x01c40018 => "FEC_OC_DTO_BURST_LEN__A",
	0x01c4001a => "FEC_OC_FCT_MODE__A",
	0x01c4001b => "FEC_OC_FCT_USAGE__A",
	0x01c4001c => "FEC_OC_FCT_OCCUPATION__A",
	0x01c4001e => "FEC_OC_TMD_MODE__A",
	0x01c4001f => "FEC_OC_TMD_COUNT__A",
	0x01c40020 => "FEC_OC_TMD_HI_MARGIN__A",
	0x01c40021 => "FEC_OC_TMD_LO_MARGIN__A",
	0x01c40022 => "FEC_OC_TMD_CTL_UPD_RATE__A",
	0x01c40023 => "FEC_OC_TMD_INT_UPD_RATE__A",
	0x01c40026 => "FEC_OC_AVR_PARM_A__A",
	0x01c40027 => "FEC_OC_AVR_PARM_B__A",
	0x01c40028 => "FEC_OC_AVR_AVG_LO__A",
	0x01c40029 => "FEC_OC_AVR_AVG_HI__A",
	0x01c4002c => "FEC_OC_RCN_MODE__A",
	0x01c4002d => "FEC_OC_RCN_OCC_SETTLE__A",
	0x01c4002e => "FEC_OC_RCN_GAIN__A",
	0x01c40030 => "FEC_OC_RCN_CTL_RATE_LO__A",
	0x01c40031 => "FEC_OC_RCN_CTL_RATE_HI__A",
	0x01c40032 => "FEC_OC_RCN_CTL_STEP_LO__A",
	0x01c40033 => "FEC_OC_RCN_CTL_STEP_HI__A",
	0x01c40034 => "FEC_OC_RCN_DTO_OFS_LO__A",
	0x01c40035 => "FEC_OC_RCN_DTO_OFS_HI__A",
	0x01c40036 => "FEC_OC_RCN_DTO_RATE_LO__A",
	0x01c40037 => "FEC_OC_RCN_DTO_RATE_HI__A",
	0x01c40038 => "FEC_OC_RCN_RATE_CLIP_LO__A",
	0x01c40039 => "FEC_OC_RCN_RATE_CLIP_HI__A",
	0x01c4003a => "FEC_OC_RCN_DYN_RATE_LO__A",
	0x01c4003b => "FEC_OC_RCN_DYN_RATE_HI__A",
	0x01c40040 => "FEC_OC_SNC_MODE__A",
	0x01c40041 => "FEC_OC_SNC_LWM__A",
	0x01c40042 => "FEC_OC_SNC_HWM__A",
	0x01c40043 => "FEC_OC_SNC_UNLOCK__A",
	0x01c40044 => "FEC_OC_SNC_LOCK_COUNT__A",
	0x01c40045 => "FEC_OC_SNC_FAIL_COUNT__A",
	0x01c40046 => "FEC_OC_SNC_FAIL_PERIOD__A",
	0x01c40047 => "FEC_OC_EMS_MODE__A",
	0x01c40048 => "FEC_OC_IPR_MODE__A",
	0x01c40049 => "FEC_OC_IPR_INVERT__A",
	0x01c40050 => "FEC_OC_OCR_MODE__A",
	0x01c40051 => "FEC_OC_OCR_RATE__A",
	0x01c40052 => "FEC_OC_OCR_INVERT__A",
	0x01c40053 => "FEC_OC_OCR_GRAB_COUNT__A",
	0x01c40054 => "FEC_OC_OCR_GRAB_SYNC__A",
	0x01c40055 => "FEC_OC_OCR_GRAB_RD0__A",
	0x01c40056 => "FEC_OC_OCR_GRAB_RD1__A",
	0x01c40057 => "FEC_OC_OCR_GRAB_RD2__A",
	0x01c40058 => "FEC_OC_OCR_GRAB_RD3__A",
	0x01c40059 => "FEC_OC_OCR_GRAB_RD4__A",
	0x01c4005a => "FEC_OC_OCR_GRAB_RD5__A",
	0x01c50000 => "FEC_DI_RAM__A",
	0x01c60000 => "FEC_RS_RAM__A",
	0x01c70000 => "FEC_OC_RAM__A",
	0x02000000 => "OFDM_FE_COMM_EXEC__A",
	0x02000001 => "OFDM_FE_COMM_STATE__A",
	0x02000002 => "OFDM_FE_COMM_MB__A",
	0x02000004 => "OFDM_FE_COMM_INT_REQ__A",
	0x02000005 => "OFDM_FE_COMM_INT_STA__A",
	0x02000006 => "OFDM_FE_COMM_INT_MSK__A",
	0x02000007 => "OFDM_FE_COMM_INT_STM__A",
	0x02010000 => "OFDM_FE_CU_COMM_EXEC__A",
	0x02010001 => "OFDM_FE_CU_COMM_STATE__A",
	0x02010002 => "OFDM_FE_CU_COMM_MB__A",
	0x02010004 => "OFDM_FE_CU_COMM_INT_REQ__A",
	0x02010005 => "OFDM_FE_CU_COMM_INT_STA__A",
	0x02010006 => "OFDM_FE_CU_COMM_INT_MSK__A",
	0x02010007 => "OFDM_FE_CU_COMM_INT_STM__A",
	0x02010010 => "OFDM_FE_CU_MODE__A",
	0x02010011 => "OFDM_FE_CU_FRM_CNT_RST__A",
	0x02010012 => "OFDM_FE_CU_FRM_CNT_STR__A",
	0x02010013 => "OFDM_FE_CU_FRM_SMP_CNT__A",
	0x02010014 => "OFDM_FE_CU_FRM_SMB_CNT__A",
	0x02010015 => "OFDM_FE_CU_CMP_MAX_DAT__A",
	0x02010016 => "OFDM_FE_CU_CMP_MAX_ADR__A",
	0x02010017 => "OFDM_FE_CU_CMP_MAX_RE__A",
	0x02010018 => "OFDM_FE_CU_CMP_MAX_IM__A",
	0x0201001f => "OFDM_FE_CU_BUF_NFC_DEL__A",
	0x02010020 => "OFDM_FE_CU_CTR_NFC_ICR__A",
	0x02010021 => "OFDM_FE_CU_CTR_NFC_OCR__A",
	0x02010022 => "OFDM_FE_CU_CTR_NFC_CNT__A",
	0x02010023 => "OFDM_FE_CU_CTR_NFC_STS__A",
	0x02010024 => "OFDM_FE_CU_DIV_NFC_REA__A",
	0x02010025 => "OFDM_FE_CU_DIV_NFC_IMA__A",
	0x02010026 => "OFDM_FE_CU_FRM_CNT_UPD__A",
	0x02010027 => "OFDM_FE_CU_DIV_NFC_CLP__A",
	0x02010028 => "OFDM_FE_CU_CMP_MAX_32__A",
	0x02010029 => "OFDM_FE_CU_CMP_MAX_16__A",
	0x0201002a => "OFDM_FE_CU_CMP_MAX_8__A",
	0x0201002b => "OFDM_FE_CU_CMP_MAX_4__A",
	0x0201002c => "OFDM_FE_CU_CMP_SUM_32_RE__A",
	0x0201002d => "OFDM_FE_CU_CMP_SUM_32_IM__A",
	0x0201002e => "OFDM_FE_CU_CMP_SUM_16_RE__A",
	0x0201002f => "OFDM_FE_CU_CMP_SUM_16_IM__A",
	0x02010030 => "OFDM_FE_CU_CMP_SUM_8_RE__A",
	0x02010031 => "OFDM_FE_CU_CMP_SUM_8_IM__A",
	0x02010032 => "OFDM_FE_CU_CMP_SUM_4_RE__A",
	0x02010033 => "OFDM_FE_CU_CMP_SUM_4_IM__A",
	0x02020000 => "OFDM_FE_CU_BUF_RAM__A",
	0x02030000 => "OFDM_FE_CU_CMP_RAM__A",
	0x02400000 => "OFDM_FT_COMM_EXEC__A",
	0x02400001 => "OFDM_FT_COMM_STATE__A",
	0x02400002 => "OFDM_FT_COMM_MB__A",
	0x02410000 => "OFDM_FT_TOP_COMM_EXEC__A",
	0x02410002 => "OFDM_FT_TOP_COMM_MB__A",
	0x02410010 => "OFDM_FT_TOP_MODE_2K__A",
	0x02410016 => "OFDM_FT_TOP_NORM_OFF__A",
	0x02420000 => "OFDM_FT_0TO2_0_RAM__A",
	0x02430000 => "OFDM_FT_0TO2_1_RAM__A",
	0x02440000 => "OFDM_FT_0TO2_2_RAM__A",
	0x02450000 => "OFDM_FT_3TO7_0_RAM__A",
	0x02460000 => "OFDM_FT_3TO7_1_RAM__A",
	0x02800000 => "OFDM_CP_COMM_EXEC__A",
	0x02800001 => "OFDM_CP_COMM_STATE__A",
	0x02800002 => "OFDM_CP_COMM_MB__A",
	0x02800004 => "OFDM_CP_COMM_INT_REQ__A",
	0x02800005 => "OFDM_CP_COMM_INT_STA__A",
	0x02800006 => "OFDM_CP_COMM_INT_MSK__A",
	0x02800007 => "OFDM_CP_COMM_INT_STM__A",
	0x02810000 => "OFDM_CP_TOP_COMM_EXEC__A",
	0x02810002 => "OFDM_CP_TOP_COMM_MB__A",
	0x02810004 => "OFDM_CP_TOP_COMM_INT_REQ__A",
	0x02810005 => "OFDM_CP_TOP_COMM_INT_STA__A",
	0x02810006 => "OFDM_CP_TOP_COMM_INT_MSK__A",
	0x02810007 => "OFDM_CP_TOP_COMM_INT_STM__A",
	0x02810010 => "OFDM_CP_TOP_MODE_2K__A",
	0x02810011 => "OFDM_CP_TOP_INTERVAL__A",
	0x02810012 => "OFDM_CP_TOP_DETECT_ENA__A",
	0x02810013 => "OFDM_CP_TOP_FIX__A",
	0x02810021 => "OFDM_CP_TOP_BR_SMB_NR__A",
	0x02810022 => "OFDM_CP_TOP_BR_CP_SMB_NR__A",
	0x02810023 => "OFDM_CP_TOP_BR_SPL_OFFSET__A",
	0x02810024 => "OFDM_CP_TOP_BR_STR_DEL__A",
	0x02810025 => "OFDM_CP_TOP_BR_EXP_ADJ__A",
	0x02810030 => "OFDM_CP_TOP_RT_ANG_INC0__A",
	0x02810031 => "OFDM_CP_TOP_RT_ANG_INC1__A",
	0x02810032 => "OFDM_CP_TOP_RT_SPD_EXP_MARG__A",
	0x02810033 => "OFDM_CP_TOP_RT_DETECT_TRH__A",
	0x02810034 => "OFDM_CP_TOP_RT_SPD_RELIABLE__A",
	0x02810035 => "OFDM_CP_TOP_RT_SPD_DIRECTION__A",
	0x02810036 => "OFDM_CP_TOP_RT_SPD_MOD__A",
	0x02810037 => "OFDM_CP_TOP_RT_SPD_SMB__A",
	0x02810038 => "OFDM_CP_TOP_RT_CPD_MODE__A",
	0x02810039 => "OFDM_CP_TOP_RT_CPD_RELIABLE__A",
	0x0281003a => "OFDM_CP_TOP_RT_CPD_BIN__A",
	0x0281003b => "OFDM_CP_TOP_RT_CPD_MAX__A",
	0x0281003c => "OFDM_CP_TOP_RT_SUPR_VAL__A",
	0x0281003d => "OFDM_CP_TOP_RT_EXP_AVE__A",
	0x0281003e => "OFDM_CP_TOP_RT_CPD_EXP_MARG__A",
	0x02810040 => "OFDM_CP_TOP_AC_NEXP_OFFS__A",
	0x02810041 => "OFDM_CP_TOP_AC_AVER_POW__A",
	0x02810042 => "OFDM_CP_TOP_AC_MAX_POW__A",
	0x02810043 => "OFDM_CP_TOP_AC_WEIGHT_MAN__A",
	0x02810044 => "OFDM_CP_TOP_AC_WEIGHT_EXP__A",
	0x02810045 => "OFDM_CP_TOP_AC_GAIN_MAN__A",
	0x02810046 => "OFDM_CP_TOP_AC_GAIN_EXP__A",
	0x02810047 => "OFDM_CP_TOP_AC_AMP_MODE__A",
	0x02810048 => "OFDM_CP_TOP_AC_AMP_FIX__A",
	0x02810049 => "OFDM_CP_TOP_AC_AMP_READ__A",
	0x0281004a => "OFDM_CP_TOP_AC_ANG_MODE__A",
	0x0281004b => "OFDM_CP_TOP_AC_ANG_OFFS__A",
	0x0281004c => "OFDM_CP_TOP_AC_ANG_READ__A",
	0x02810050 => "OFDM_CP_TOP_DL_MB_WR_ADDR__A",
	0x02810051 => "OFDM_CP_TOP_DL_MB_WR_CTR__A",
	0x02810052 => "OFDM_CP_TOP_DL_MB_RD_ADDR__A",
	0x02810053 => "OFDM_CP_TOP_DL_MB_RD_CTR__A",
	0x02810060 => "OFDM_CP_TOP_AC_ACCU_REAL0__A",
	0x02810061 => "OFDM_CP_TOP_AC_ACCU_IMAG0__A",
	0x02810062 => "OFDM_CP_TOP_AC_ACCU_REAL1__A",
	0x02810063 => "OFDM_CP_TOP_AC_ACCU_IMAG1__A",
	0x02820000 => "OFDM_CP_BR_BUF_CPL_RAM__A",
	0x02830000 => "OFDM_CP_BR_BUF_DAT_RAM__A",
	0x02840000 => "OFDM_CP_DL_0_RAM__A",
	0x02850000 => "OFDM_CP_DL_1_RAM__A",
	0x02860000 => "OFDM_CP_DL_2_RAM__A",
	0x02c00000 => "OFDM_CE_COMM_EXEC__A",
	0x02c00001 => "OFDM_CE_COMM_STATE__A",
	0x02c00002 => "OFDM_CE_COMM_MB__A",
	0x02c00004 => "OFDM_CE_COMM_INT_REQ__A",
	0x02c00005 => "OFDM_CE_COMM_INT_STA__A",
	0x02c00006 => "OFDM_CE_COMM_INT_MSK__A",
	0x02c00007 => "OFDM_CE_COMM_INT_STM__A",
	0x02c10000 => "OFDM_CE_TOP_COMM_EXEC__A",
	0x02c10002 => "OFDM_CE_TOP_COMM_MB__A",
	0x02c10004 => "OFDM_CE_TOP_COMM_INT_REQ__A",
	0x02c10005 => "OFDM_CE_TOP_COMM_INT_STA__A",
	0x02c10006 => "OFDM_CE_TOP_COMM_INT_MSK__A",
	0x02c10007 => "OFDM_CE_TOP_COMM_INT_STM__A",
	0x02c10010 => "OFDM_CE_TOP_MODE_2K__A",
	0x02c10011 => "OFDM_CE_TOP_TAPSET__A",
	0x02c10012 => "OFDM_CE_TOP_AVG_POW__A",
	0x02c10013 => "OFDM_CE_TOP_MAX_POW__A",
	0x02c10014 => "OFDM_CE_TOP_ATT__A",
	0x02c10015 => "OFDM_CE_TOP_NRED__A",
	0x02c10020 => "OFDM_CE_TOP_PU_SIGN__A",
	0x02c10021 => "OFDM_CE_TOP_PU_MIX__A",
	0x02c10030 => "OFDM_CE_TOP_PB_PILOT_REQ__A",
	0x02c10031 => "OFDM_CE_TOP_PB_PILOT_REQ_VALID__A",
	0x02c10032 => "OFDM_CE_TOP_PB_FREEZE__A",
	0x02c10038 => "OFDM_CE_TOP_PB_PILOT_EXP__A",
	0x02c10039 => "OFDM_CE_TOP_PB_PILOT_REAL__A",
	0x02c1003a => "OFDM_CE_TOP_PB_PILOT_IMAG__A",
	0x02c1003b => "OFDM_CE_TOP_PB_SMBNR__A",
	0x02c10040 => "OFDM_CE_TOP_NE_PILOT_REQ__A",
	0x02c10041 => "OFDM_CE_TOP_NE_PILOT_REQ_VALID__A",
	0x02c10042 => "OFDM_CE_TOP_NE_PILOT_DATA__A",
	0x02c10043 => "OFDM_CE_TOP_NE_ERR_SELECT__A",
	0x02c10044 => "OFDM_CE_TOP_NE_TD_CAL__A",
	0x02c10045 => "OFDM_CE_TOP_NE_FD_CAL__A",
	0x02c10046 => "OFDM_CE_TOP_NE_MIXAVG__A",
	0x02c10047 => "OFDM_CE_TOP_NE_NUPD_OFS__A",
	0x02c10048 => "OFDM_CE_TOP_NE_TD_POW__A",
	0x02c10049 => "OFDM_CE_TOP_NE_FD_POW__A",
	0x02c1004a => "OFDM_CE_TOP_NE_NEXP_AVG__A",
	0x02c1004b => "OFDM_CE_TOP_NE_OFFSET__A",
	0x02c1004c => "OFDM_CE_TOP_NE_NUPD_TRH__A",
	0x02c10050 => "OFDM_CE_TOP_PE_NEXP_OFFS__A",
	0x02c10051 => "OFDM_CE_TOP_PE_TIMESHIFT__A",
	0x02c10052 => "OFDM_CE_TOP_PE_DIF_REAL_L__A",
	0x02c10053 => "OFDM_CE_TOP_PE_DIF_IMAG_L__A",
	0x02c10054 => "OFDM_CE_TOP_PE_DIF_REAL_R__A",
	0x02c10055 => "OFDM_CE_TOP_PE_DIF_IMAG_R__A",
	0x02c10056 => "OFDM_CE_TOP_PE_ABS_REAL_L__A",
	0x02c10057 => "OFDM_CE_TOP_PE_ABS_IMAG_L__A",
	0x02c10058 => "OFDM_CE_TOP_PE_ABS_REAL_R__A",
	0x02c10059 => "OFDM_CE_TOP_PE_ABS_IMAG_R__A",
	0x02c1005a => "OFDM_CE_TOP_PE_ABS_EXP_L__A",
	0x02c1005b => "OFDM_CE_TOP_PE_ABS_EXP_R__A",
	0x02c10060 => "OFDM_CE_TOP_TP_UPDATE_MODE__A",
	0x02c10061 => "OFDM_CE_TOP_TP_LMS_TAP_ON__A",
	0x02c10064 => "OFDM_CE_TOP_TP_A0_TAP_NEW__A",
	0x02c10065 => "OFDM_CE_TOP_TP_A0_TAP_NEW_VALID__A",
	0x02c10066 => "OFDM_CE_TOP_TP_A0_MU_LMS_STEP__A",
	0x02c10067 => "OFDM_CE_TOP_TP_A0_TAP_CURR__A",
	0x02c10068 => "OFDM_CE_TOP_TP_A1_TAP_NEW__A",
	0x02c10069 => "OFDM_CE_TOP_TP_A1_TAP_NEW_VALID__A",
	0x02c1006a => "OFDM_CE_TOP_TP_A1_MU_LMS_STEP__A",
	0x02c1006b => "OFDM_CE_TOP_TP_A1_TAP_CURR__A",
	0x02c1006c => "OFDM_CE_TOP_TP_DOPP_ENERGY__A",
	0x02c1006d => "OFDM_CE_TOP_TP_DOPP_DIFF_ENERGY__A",
	0x02c1006e => "OFDM_CE_TOP_TP_A0_TAP_ENERGY__A",
	0x02c1006f => "OFDM_CE_TOP_TP_A1_TAP_ENERGY__A",
	0x02c10072 => "OFDM_CE_TOP_TI_SYM_CNT__A",
	0x02c10073 => "OFDM_CE_TOP_TI_PHN_ENABLE__A",
	0x02c10074 => "OFDM_CE_TOP_TI_SHIFT__A",
	0x02c10075 => "OFDM_CE_TOP_TI_SLOW__A",
	0x02c10076 => "OFDM_CE_TOP_TI_MGAIN__A",
	0x02c10077 => "OFDM_CE_TOP_TI_ACCU1__A",
	0x02c10090 => "OFDM_CE_TOP_FI_SHT_INCR__A",
	0x02c10091 => "OFDM_CE_TOP_FI_EXP_NORM__A",
	0x02c10092 => "OFDM_CE_TOP_FI_SUPR_VAL__A",
	0x02c100a0 => "OFDM_CE_TOP_IR_INPUTSEL__A",
	0x02c100a1 => "OFDM_CE_TOP_IR_STARTPOS__A",
	0x02c100a2 => "OFDM_CE_TOP_IR_NEXP_THRES__A",
	0x02c100a3 => "OFDM_CE_TOP_IR_LENGTH__A",
	0x02c100a4 => "OFDM_CE_TOP_IR_FREQ__A",
	0x02c100a5 => "OFDM_CE_TOP_IR_FREQINC__A",
	0x02c100a6 => "OFDM_CE_TOP_IR_KAISINC__A",
	0x02c100a7 => "OFDM_CE_TOP_IR_CTL__A",
	0x02c100a8 => "OFDM_CE_TOP_IR_REAL__A",
	0x02c100a9 => "OFDM_CE_TOP_IR_IMAG__A",
	0x02c100aa => "OFDM_CE_TOP_IR_INDEX__A",
	0x02c100b0 => "OFDM_CE_TOP_NI_PER_LEFT__A",
	0x02c100b1 => "OFDM_CE_TOP_NI_PER_RIGHT__A",
	0x02c100b2 => "OFDM_CE_TOP_NI_POS_LR__A",
	0x02c20000 => "OFDM_CE_FR_COMM_EXEC__A",
	0x02c20010 => "OFDM_CE_FR_TREAL00__A",
	0x02c20011 => "OFDM_CE_FR_TIMAG00__A",
	0x02c20012 => "OFDM_CE_FR_TREAL01__A",
	0x02c20013 => "OFDM_CE_FR_TIMAG01__A",
	0x02c20014 => "OFDM_CE_FR_TREAL02__A",
	0x02c20015 => "OFDM_CE_FR_TIMAG02__A",
	0x02c20016 => "OFDM_CE_FR_TREAL03__A",
	0x02c20017 => "OFDM_CE_FR_TIMAG03__A",
	0x02c20018 => "OFDM_CE_FR_TREAL04__A",
	0x02c20019 => "OFDM_CE_FR_TIMAG04__A",
	0x02c2001a => "OFDM_CE_FR_TREAL05__A",
	0x02c2001b => "OFDM_CE_FR_TIMAG05__A",
	0x02c2001c => "OFDM_CE_FR_TREAL06__A",
	0x02c2001d => "OFDM_CE_FR_TIMAG06__A",
	0x02c2001e => "OFDM_CE_FR_TREAL07__A",
	0x02c2001f => "OFDM_CE_FR_TIMAG07__A",
	0x02c20020 => "OFDM_CE_FR_TREAL08__A",
	0x02c20021 => "OFDM_CE_FR_TIMAG08__A",
	0x02c20022 => "OFDM_CE_FR_TREAL09__A",
	0x02c20023 => "OFDM_CE_FR_TIMAG09__A",
	0x02c20024 => "OFDM_CE_FR_TREAL10__A",
	0x02c20025 => "OFDM_CE_FR_TIMAG10__A",
	0x02c20026 => "OFDM_CE_FR_TREAL11__A",
	0x02c20027 => "OFDM_CE_FR_TIMAG11__A",
	0x02c20028 => "OFDM_CE_FR_MID_TAP__A",
	0x02c20029 => "OFDM_CE_FR_SQS_G00__A",
	0x02c2002a => "OFDM_CE_FR_SQS_G01__A",
	0x02c2002b => "OFDM_CE_FR_SQS_G02__A",
	0x02c2002c => "OFDM_CE_FR_SQS_G03__A",
	0x02c2002d => "OFDM_CE_FR_SQS_G04__A",
	0x02c2002e => "OFDM_CE_FR_SQS_G05__A",
	0x02c2002f => "OFDM_CE_FR_SQS_G06__A",
	0x02c20030 => "OFDM_CE_FR_SQS_G07__A",
	0x02c20031 => "OFDM_CE_FR_SQS_G08__A",
	0x02c20032 => "OFDM_CE_FR_SQS_G09__A",
	0x02c20033 => "OFDM_CE_FR_SQS_G10__A",
	0x02c20034 => "OFDM_CE_FR_SQS_G11__A",
	0x02c20035 => "OFDM_CE_FR_SQS_G12__A",
	0x02c20036 => "OFDM_CE_FR_RIO_G00__A",
	0x02c20037 => "OFDM_CE_FR_RIO_G01__A",
	0x02c20038 => "OFDM_CE_FR_RIO_G02__A",
	0x02c20039 => "OFDM_CE_FR_RIO_G03__A",
	0x02c2003a => "OFDM_CE_FR_RIO_G04__A",
	0x02c2003b => "OFDM_CE_FR_RIO_G05__A",
	0x02c2003c => "OFDM_CE_FR_RIO_G06__A",
	0x02c2003d => "OFDM_CE_FR_RIO_G07__A",
	0x02c2003e => "OFDM_CE_FR_RIO_G08__A",
	0x02c2003f => "OFDM_CE_FR_RIO_G09__A",
	0x02c20040 => "OFDM_CE_FR_RIO_G10__A",
	0x02c20041 => "OFDM_CE_FR_MODE__A",
	0x02c20042 => "OFDM_CE_FR_SQS_TRH__A",
	0x02c20043 => "OFDM_CE_FR_RIO_GAIN__A",
	0x02c20044 => "OFDM_CE_FR_BYPASS__A",
	0x02c20045 => "OFDM_CE_FR_PM_SET__A",
	0x02c20046 => "OFDM_CE_FR_ERR_SH__A",
	0x02c20047 => "OFDM_CE_FR_MAN_SH__A",
	0x02c20048 => "OFDM_CE_FR_TAP_SH__A",
	0x02c20049 => "OFDM_CE_FR_CLIP__A",
	0x02c2004a => "OFDM_CE_FR_LEAK_UPD__A",
	0x02c2004b => "OFDM_CE_FR_LEAK_SH__A",
	0x02c30000 => "OFDM_CE_NE_RAM__A",
	0x02c40000 => "OFDM_CE_PB_RAM__A",
	0x03000000 => "OFDM_EQ_COMM_EXEC__A",
	0x03000001 => "OFDM_EQ_COMM_STATE__A",
	0x03000002 => "OFDM_EQ_COMM_MB__A",
	0x03000004 => "OFDM_EQ_COMM_INT_REQ__A",
	0x03000005 => "OFDM_EQ_COMM_INT_STA__A",
	0x03000006 => "OFDM_EQ_COMM_INT_MSK__A",
	0x03000007 => "OFDM_EQ_COMM_INT_STM__A",
	0x03010000 => "OFDM_EQ_TOP_COMM_EXEC__A",
	0x03010001 => "OFDM_EQ_TOP_COMM_STATE__A",
	0x03010002 => "OFDM_EQ_TOP_COMM_MB__A",
	0x03010004 => "OFDM_EQ_TOP_COMM_INT_REQ__A",
	0x03010005 => "OFDM_EQ_TOP_COMM_INT_STA__A",
	0x03010006 => "OFDM_EQ_TOP_COMM_INT_MSK__A",
	0x03010007 => "OFDM_EQ_TOP_COMM_INT_STM__A",
	0x03010014 => "OFDM_EQ_TOP_IS_MODE__A",
	0x03010015 => "OFDM_EQ_TOP_IS_GAIN_MAN__A",
	0x03010016 => "OFDM_EQ_TOP_IS_GAIN_EXP__A",
	0x03010017 => "OFDM_EQ_TOP_IS_CLIP_EXP__A",
	0x0301001e => "OFDM_EQ_TOP_DV_MODE__A",
	0x0301001f => "OFDM_EQ_TOP_DV_POS_CLIP_DAT__A",
	0x03010028 => "OFDM_EQ_TOP_SN_MODE__A",
	0x03010029 => "OFDM_EQ_TOP_SN_PFIX__A",
	0x0301002a => "OFDM_EQ_TOP_SN_CEGAIN__A",
	0x0301002b => "OFDM_EQ_TOP_SN_OFFSET__A",
	0x0301002c => "OFDM_EQ_TOP_SN_NULLIFY__A",
	0x0301002d => "OFDM_EQ_TOP_SN_SQUASH__A",
	0x03010032 => "OFDM_EQ_TOP_RC_SEL_CAR__A",
	0x03010033 => "OFDM_EQ_TOP_RC_STS__A",
	0x03010046 => "OFDM_EQ_TOP_OT_CONST__A",
	0x03010047 => "OFDM_EQ_TOP_OT_ALPHA__A",
	0x03010048 => "OFDM_EQ_TOP_OT_QNT_THRES0__A",
	0x03010049 => "OFDM_EQ_TOP_OT_QNT_THRES1__A",
	0x0301004a => "OFDM_EQ_TOP_OT_CSI_STEP__A",
	0x0301004b => "OFDM_EQ_TOP_OT_CSI_OFFSET__A",
	0x0301004c => "OFDM_EQ_TOP_OT_CSI_GAIN__A",
	0x0301004d => "OFDM_EQ_TOP_OT_CSI_MEAN__A",
	0x0301004e => "OFDM_EQ_TOP_OT_CSI_VARIANCE__A",
	0x03010050 => "OFDM_EQ_TOP_TD_TPS_INIT__A",
	0x03010051 => "OFDM_EQ_TOP_TD_TPS_SYNC__A",
	0x03010052 => "OFDM_EQ_TOP_TD_TPS_LEN__A",
	0x03010053 => "OFDM_EQ_TOP_TD_TPS_FRM_NMB__A",
	0x03010054 => "OFDM_EQ_TOP_TD_TPS_CONST__A",
	0x03010055 => "OFDM_EQ_TOP_TD_TPS_HINFO__A",
	0x03010056 => "OFDM_EQ_TOP_TD_TPS_CODE_HP__A",
	0x03010057 => "OFDM_EQ_TOP_TD_TPS_CODE_LP__A",
	0x03010058 => "OFDM_EQ_TOP_TD_TPS_GUARD__A",
	0x03010059 => "OFDM_EQ_TOP_TD_TPS_TR_MODE__A",
	0x0301005a => "OFDM_EQ_TOP_TD_TPS_CELL_ID_HI__A",
	0x0301005b => "OFDM_EQ_TOP_TD_TPS_CELL_ID_LO__A",
	0x0301005c => "OFDM_EQ_TOP_TD_TPS_RSV__A",
	0x0301005d => "OFDM_EQ_TOP_TD_TPS_BCH__A",
	0x0301005e => "OFDM_EQ_TOP_TD_SQR_ERR_I__A",
	0x0301005f => "OFDM_EQ_TOP_TD_SQR_ERR_Q__A",
	0x03010060 => "OFDM_EQ_TOP_TD_SQR_ERR_EXP__A",
	0x03010061 => "OFDM_EQ_TOP_TD_REQ_SMB_CNT__A",
	0x03010062 => "OFDM_EQ_TOP_TD_TPS_PWR_OFS__A",
	0x03400000 => "OFDM_EC_COMM_EXEC__A",
	0x03400001 => "OFDM_EC_COMM_STATE__A",
	0x03400002 => "OFDM_EC_COMM_MB__A",
	0x03400004 => "OFDM_EC_COMM_INT_REQ__A",
	0x03400005 => "OFDM_EC_COMM_INT_STA__A",
	0x03400006 => "OFDM_EC_COMM_INT_MSK__A",
	0x03400007 => "OFDM_EC_COMM_INT_STM__A",
	0x03410000 => "OFDM_EC_SB_COMM_EXEC__A",
	0x03410001 => "OFDM_EC_SB_COMM_STATE__A",
	0x03410002 => "OFDM_EC_SB_COMM_MB__A",
	0x03410010 => "OFDM_EC_SB_TR_MODE__A",
	0x03410011 => "OFDM_EC_SB_CONST__A",
	0x03410012 => "OFDM_EC_SB_ALPHA__A",
	0x03410013 => "OFDM_EC_SB_PRIOR__A",
	0x03410014 => "OFDM_EC_SB_CSI_HI__A",
	0x03410015 => "OFDM_EC_SB_CSI_LO__A",
	0x03410016 => "OFDM_EC_SB_SMB_TGL__A",
	0x03410017 => "OFDM_EC_SB_SNR_HI__A",
	0x03410018 => "OFDM_EC_SB_SNR_MID__A",
	0x03410019 => "OFDM_EC_SB_SNR_LO__A",
	0x0341001a => "OFDM_EC_SB_SCALE_MSB__A",
	0x0341001b => "OFDM_EC_SB_SCALE_BIT2__A",
	0x0341001c => "OFDM_EC_SB_SCALE_LSB__A",
	0x0341001d => "OFDM_EC_SB_CSI_OFS0__A",
	0x0341001e => "OFDM_EC_SB_CSI_OFS1__A",
	0x0341001f => "OFDM_EC_SB_CSI_OFS2__A",
	0x03410020 => "OFDM_EC_SB_MAX0__A",
	0x03410021 => "OFDM_EC_SB_MAX1__A",
	0x03410022 => "OFDM_EC_SB_MAX2__A",
	0x03410023 => "OFDM_EC_SB_CSI_DIS__A",
	0x03420000 => "OFDM_EC_VD_COMM_EXEC__A",
	0x03420001 => "OFDM_EC_VD_COMM_STATE__A",
	0x03420002 => "OFDM_EC_VD_COMM_MB__A",
	0x03420003 => "OFDM_EC_VD_COMM_INT_REQ__A",
	0x03420005 => "OFDM_EC_VD_COMM_INT_STA__A",
	0x03420006 => "OFDM_EC_VD_COMM_INT_MSK__A",
	0x03420007 => "OFDM_EC_VD_COMM_INT_STM__A",
	0x03420010 => "OFDM_EC_VD_FORCE__A",
	0x03420011 => "OFDM_EC_VD_SET_CODERATE__A",
	0x03420012 => "OFDM_EC_VD_REQ_SMB_CNT__A",
	0x03420013 => "OFDM_EC_VD_REQ_BIT_CNT__A",
	0x03420014 => "OFDM_EC_VD_RLK_ENA__A",
	0x03420015 => "OFDM_EC_VD_VAL__A",
	0x03420016 => "OFDM_EC_VD_GET_CODERATE__A",
	0x03420017 => "OFDM_EC_VD_ERR_BIT_CNT__A",
	0x03420018 => "OFDM_EC_VD_IN_BIT_CNT__A",
	0x03420019 => "OFDM_EC_VD_STS__A",
	0x0342001a => "OFDM_EC_VD_RLK_CNT__A",
	0x03430000 => "OFDM_EC_SY_COMM_EXEC__A",
	0x03430002 => "OFDM_EC_SY_COMM_MB__A",
	0x03430003 => "OFDM_EC_SY_COMM_INT_REQ__A",
	0x03430005 => "OFDM_EC_SY_COMM_INT_STA__A",
	0x03430006 => "OFDM_EC_SY_COMM_INT_MSK__A",
	0x03430007 => "OFDM_EC_SY_COMM_INT_STM__A",
	0x03430010 => "OFDM_EC_SY_STATUS__A",
	0x03430011 => "OFDM_EC_SY_TIMEOUT__A",
	0x03430012 => "OFDM_EC_SY_SYNC_LWM__A",
	0x03430013 => "OFDM_EC_SY_SYNC_AWM__A",
	0x03430014 => "OFDM_EC_SY_SYNC_HWM__A",
	0x03430015 => "OFDM_EC_SY_UNLOCK__A",
	0x03440000 => "OFDM_EC_SB_BD0_RAM__A",
	0x03450000 => "OFDM_EC_SB_BD1_RAM__A",
	0x03460000 => "OFDM_EC_SB_SD_RAM__A",
	0x03470000 => "OFDM_EC_VD_RE_RAM__A",
	0x03480000 => "OFDM_EC_VD_TB0_RAM__A",
	0x03490000 => "OFDM_EC_VD_TB1_RAM__A",
	0x034a0000 => "OFDM_EC_VD_TB2_RAM__A",
	0x034b0000 => "OFDM_EC_VD_TB3_RAM__A",
	0x03800000 => "OFDM_LC_COMM_EXEC__A",
	0x03800001 => "OFDM_LC_COMM_STATE__A",
	0x03800002 => "OFDM_LC_COMM_MB__A",
	0x03800004 => "OFDM_LC_COMM_INT_REQ__A",
	0x03800005 => "OFDM_LC_COMM_INT_STA__A",
	0x03800006 => "OFDM_LC_COMM_INT_MSK__A",
	0x03800007 => "OFDM_LC_COMM_INT_STM__A",
	0x03810000 => "OFDM_LC_CT_COMM_EXEC__A",
	0x03810001 => "OFDM_LC_CT_COMM_STATE__A",
	0x03810004 => "OFDM_LC_CT_COMM_INT_REQ__A",
	0x03810005 => "OFDM_LC_CT_COMM_INT_STA__A",
	0x03810006 => "OFDM_LC_CT_COMM_INT_MSK__A",
	0x03810007 => "OFDM_LC_CT_COMM_INT_STM__A",
	0x03810010 => "OFDM_LC_CT_CTL_STK_0__A",
	0x03810011 => "OFDM_LC_CT_CTL_STK_1__A",
	0x03810012 => "OFDM_LC_CT_CTL_STK_2__A",
	0x03810013 => "OFDM_LC_CT_CTL_STK_3__A",
	0x0381001f => "OFDM_LC_CT_CTL_BPT_IDX__A",
	0x03810020 => "OFDM_LC_CT_CTL_BPT__A",
	0x03820000 => "OFDM_LC_RA_RAM__A",
	0x03820006 => "OFDM_LC_RA_RAM_PROC_DELAY_IF__A",
	0x03820007 => "OFDM_LC_RA_RAM_PROC_DELAY_FS__A",
	0x03820008 => "OFDM_LC_RA_RAM_LOCK_TH_CRMM__A",
	0x03820009 => "OFDM_LC_RA_RAM_LOCK_TH_SRMM__A",
	0x0382000a => "OFDM_LC_RA_RAM_LOCK_COUNT__A",
	0x0382000b => "OFDM_LC_RA_RAM_CPRTOFS_NOM__A",
	0x0382000c => "OFDM_LC_RA_RAM_IFINCR_NOM_L__A",
	0x0382000d => "OFDM_LC_RA_RAM_IFINCR_NOM_H__A",
	0x0382000e => "OFDM_LC_RA_RAM_FSINCR_NOM_L__A",
	0x0382000f => "OFDM_LC_RA_RAM_FSINCR_NOM_H__A",
	0x03820010 => "OFDM_LC_RA_RAM_MODE_2K__A",
	0x03820011 => "OFDM_LC_RA_RAM_MODE_GUARD__A",
	0x03820012 => "OFDM_LC_RA_RAM_MODE_ADJUST__A",
	0x03820014 => "OFDM_LC_RA_RAM_RC_STS__A",
	0x03820018 => "OFDM_LC_RA_RAM_ACTUAL_CP_DIF_CRMM__A",
	0x03820019 => "OFDM_LC_RA_RAM_ACTUAL_CP_DIF_SRMM__A",
	0x0382001a => "OFDM_LC_RA_RAM_FILTER_SYM_SET__A",
	0x0382001b => "OFDM_LC_RA_RAM_FILTER_SYM_CUR__A",
	0x0382001c => "OFDM_LC_RA_RAM_DIVERSITY_DELAY__A",
	0x0382001d => "OFDM_LC_RA_RAM_MAX_ABS_EXP__A",
	0x0382001f => "OFDM_LC_RA_RAM_ACTUAL_CP_CRMM__A",
	0x03820020 => "OFDM_LC_RA_RAM_ACTUAL_CE_CRMM__A",
	0x03820021 => "OFDM_LC_RA_RAM_ACTUAL_CE_SRMM__A",
	0x03820022 => "OFDM_LC_RA_RAM_ACTUAL_PHASE__A",
	0x03820023 => "OFDM_LC_RA_RAM_ACTUAL_DELAY__A",
	0x03820024 => "OFDM_LC_RA_RAM_ADJUST_CRMM__A",
	0x03820025 => "OFDM_LC_RA_RAM_ADJUST_SRMM__A",
	0x03820026 => "OFDM_LC_RA_RAM_ADJUST_PHASE__A",
	0x03820027 => "OFDM_LC_RA_RAM_ADJUST_DELAY__A",
	0x03820028 => "OFDM_LC_RA_RAM_PIPE_CP_PHASE_0__A",
	0x03820029 => "OFDM_LC_RA_RAM_PIPE_CP_PHASE_1__A",
	0x0382002a => "OFDM_LC_RA_RAM_PIPE_CP_PHASE_CON__A",
	0x0382002b => "OFDM_LC_RA_RAM_PIPE_CP_PHASE_DIF__A",
	0x0382002c => "OFDM_LC_RA_RAM_PIPE_CP_PHASE_RES__A",
	0x0382002d => "OFDM_LC_RA_RAM_PIPE_CP_PHASE_RZ__A",
	0x0382002e => "OFDM_LC_RA_RAM_FILTER_BACKUP__A",
	0x03820030 => "OFDM_LC_RA_RAM_PIPE_CP_CRMM_0__A",
	0x03820031 => "OFDM_LC_RA_RAM_PIPE_CP_CRMM_1__A",
	0x03820032 => "OFDM_LC_RA_RAM_PIPE_CP_CRMM_CON__A",
	0x03820033 => "OFDM_LC_RA_RAM_PIPE_CP_CRMM_DIF__A",
	0x03820034 => "OFDM_LC_RA_RAM_PIPE_CP_CRMM_RES__A",
	0x03820035 => "OFDM_LC_RA_RAM_PIPE_CP_CRMM_RZ__A",
	0x03820038 => "OFDM_LC_RA_RAM_PIPE_CP_SRMM_0__A",
	0x03820039 => "OFDM_LC_RA_RAM_PIPE_CP_SRMM_1__A",
	0x0382003a => "OFDM_LC_RA_RAM_PIPE_CP_SRMM_CON__A",
	0x0382003b => "OFDM_LC_RA_RAM_PIPE_CP_SRMM_DIF__A",
	0x0382003c => "OFDM_LC_RA_RAM_PIPE_CP_SRMM_RES__A",
	0x0382003d => "OFDM_LC_RA_RAM_PIPE_CP_SRMM_RZ__A",
	0x03820060 => "OFDM_LC_RA_RAM_FILTER_CRMM_A__A",
	0x03820061 => "OFDM_LC_RA_RAM_FILTER_CRMM_B__A",
	0x03820062 => "OFDM_LC_RA_RAM_FILTER_CRMM_Z1_0__A",
	0x03820063 => "OFDM_LC_RA_RAM_FILTER_CRMM_Z1_1__A",
	0x03820064 => "OFDM_LC_RA_RAM_FILTER_CRMM_Z2_0__A",
	0x03820065 => "OFDM_LC_RA_RAM_FILTER_CRMM_Z2_1__A",
	0x03820066 => "OFDM_LC_RA_RAM_FILTER_CRMM_TMP_0__A",
	0x03820067 => "OFDM_LC_RA_RAM_FILTER_CRMM_TMP_1__A",
	0x03820068 => "OFDM_LC_RA_RAM_FILTER_SRMM_A__A",
	0x03820069 => "OFDM_LC_RA_RAM_FILTER_SRMM_B__A",
	0x0382006a => "OFDM_LC_RA_RAM_FILTER_SRMM_Z1_0__A",
	0x0382006b => "OFDM_LC_RA_RAM_FILTER_SRMM_Z1_1__A",
	0x0382006c => "OFDM_LC_RA_RAM_FILTER_SRMM_Z2_0__A",
	0x0382006d => "OFDM_LC_RA_RAM_FILTER_SRMM_Z2_1__A",
	0x0382006e => "OFDM_LC_RA_RAM_FILTER_SRMM_TMP_0__A",
	0x0382006f => "OFDM_LC_RA_RAM_FILTER_SRMM_TMP_1__A",
	0x03820070 => "OFDM_LC_RA_RAM_FILTER_PHASE_A__A",
	0x03820071 => "OFDM_LC_RA_RAM_FILTER_PHASE_B__A",
	0x03820072 => "OFDM_LC_RA_RAM_FILTER_PHASE_Z1_0__A",
	0x03820073 => "OFDM_LC_RA_RAM_FILTER_PHASE_Z1_1__A",
	0x03820074 => "OFDM_LC_RA_RAM_FILTER_PHASE_Z2_0__A",
	0x03820075 => "OFDM_LC_RA_RAM_FILTER_PHASE_Z2_1__A",
	0x03820076 => "OFDM_LC_RA_RAM_FILTER_PHASE_TMP_0__A",
	0x03820077 => "OFDM_LC_RA_RAM_FILTER_PHASE_TMP_1__A",
	0x03820078 => "OFDM_LC_RA_RAM_FILTER_DELAY_A__A",
	0x03820079 => "OFDM_LC_RA_RAM_FILTER_DELAY_B__A",
	0x0382007a => "OFDM_LC_RA_RAM_FILTER_DELAY_Z1_0__A",
	0x0382007b => "OFDM_LC_RA_RAM_FILTER_DELAY_Z1_1__A",
	0x0382007c => "OFDM_LC_RA_RAM_FILTER_DELAY_Z2_0__A",
	0x0382007d => "OFDM_LC_RA_RAM_FILTER_DELAY_Z2_1__A",
	0x0382007e => "OFDM_LC_RA_RAM_FILTER_DELAY_TMP_0__A",
	0x0382007f => "OFDM_LC_RA_RAM_FILTER_DELAY_TMP_1__A",
	0x03830000 => "OFDM_LC_IF_RAM_TRP_BPT0_0__A",
	0x03830001 => "OFDM_LC_IF_RAM_TRP_BPT0_1__A",
	0x03830002 => "OFDM_LC_IF_RAM_TRP_STKU_0__A",
	0x03830004 => "OFDM_LC_IF_RAM_TRP_STKU_1__A",
	0x03830006 => "OFDM_LC_IF_RAM_TRP_WARM_0__A",
	0x03830007 => "OFDM_LC_IF_RAM_TRP_WARM_1__A",
	0x03c00000 => "OFDM_SC_COMM_EXEC__A",
	0x03c00001 => "OFDM_SC_COMM_STATE__A",
	0x03c00002 => "OFDM_SC_COMM_MB__A",
	0x03c00004 => "OFDM_SC_COMM_INT_REQ__A",
	0x03c00005 => "OFDM_SC_COMM_INT_STA__A",
	0x03c00006 => "OFDM_SC_COMM_INT_MSK__A",
	0x03c00007 => "OFDM_SC_COMM_INT_STM__A",
	0x03c10000 => "OFDM_SC_CT_COMM_EXEC__A",
	0x03c10001 => "OFDM_SC_CT_COMM_STATE__A",
	0x03c10004 => "OFDM_SC_CT_COMM_INT_REQ__A",
	0x03c10005 => "OFDM_SC_CT_COMM_INT_STA__A",
	0x03c10006 => "OFDM_SC_CT_COMM_INT_MSK__A",
	0x03c10007 => "OFDM_SC_CT_COMM_INT_STM__A",
	0x03c10010 => "OFDM_SC_CT_CTL_STK_0__A",
	0x03c10011 => "OFDM_SC_CT_CTL_STK_1__A",
	0x03c10012 => "OFDM_SC_CT_CTL_STK_2__A",
	0x03c10013 => "OFDM_SC_CT_CTL_STK_3__A",
	0x03c1001f => "OFDM_SC_CT_CTL_BPT_IDX__A",
	0x03c10020 => "OFDM_SC_CT_CTL_BPT__A",
	0x03c20000 => "OFDM_SC_RA_RAM__A",
	0x03c20040 => "OFDM_SC_RA_RAM_PARAM0__A",
	0x03c20041 => "OFDM_SC_RA_RAM_PARAM1__A",
	0x03c20042 => "OFDM_SC_RA_RAM_CMD_ADDR__A",
	0x03c20043 => "OFDM_SC_RA_RAM_CMD__A",
	0x03c20044 => "OFDM_SC_RA_RAM_PROC_ACTIVATE__A",
	0x03c20045 => "OFDM_SC_RA_RAM_PROC_TERMINATED__A",
	0x03c20046 => "OFDM_SC_RA_RAM_SW_EVENT__A",
	0x03c20047 => "OFDM_SC_RA_RAM_LOCKTRACK__A",
	0x03c20048 => "OFDM_SC_RA_RAM_OP_PARAM__A",
	0x03c20049 => "OFDM_SC_RA_RAM_OP_AUTO__A",
	0x03c2004a => "OFDM_SC_RA_RAM_PILOT_STATUS__A",
	0x03c2004b => "OFDM_SC_RA_RAM_LOCK__A",
	0x03c2004c => "OFDM_SC_RA_RAM_BE_OPT_ENA__A",
	0x03c2004d => "OFDM_SC_RA_RAM_BE_OPT_DELAY__A",
	0x03c2004e => "OFDM_SC_RA_RAM_BE_OPT_INIT_DELAY__A",
	0x03c2004f => "OFDM_SC_RA_RAM_ECHO_THRES__A",
	0x03c20050 => "OFDM_SC_RA_RAM_CONFIG__A",
	0x03c20054 => "OFDM_SC_RA_RAM_CE_REG_NE_FD_OFF__A",
	0x03c20055 => "OFDM_SC_RA_RAM_FR_2K_MAN_SH__A",
	0x03c20056 => "OFDM_SC_RA_RAM_FR_2K_TAP_SH__A",
	0x03c20057 => "OFDM_SC_RA_RAM_FR_2K_LEAK_UPD__A",
	0x03c20058 => "OFDM_SC_RA_RAM_FR_2K_LEAK_SH__A",
	0x03c20059 => "OFDM_SC_RA_RAM_FR_8K_MAN_SH__A",
	0x03c2005a => "OFDM_SC_RA_RAM_FR_8K_TAP_SH__A",
	0x03c2005b => "OFDM_SC_RA_RAM_FR_8K_LEAK_UPD__A",
	0x03c2005c => "OFDM_SC_RA_RAM_FR_8K_LEAK_SH__A",
	0x03c2005d => "OFDM_SC_RA_RAM_CO_TD_CAL_2K__A",
	0x03c2005e => "OFDM_SC_RA_RAM_CO_TD_CAL_8K__A",
	0x03c2005f => "OFDM_SC_RA_RAM_MOTION_OFFSET__A",
	0x03c20060 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_1__A",
	0x03c20061 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_2__A",
	0x03c20062 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_3__A",
	0x03c20063 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_4__A",
	0x03c20064 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_5__A",
	0x03c20065 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_6__A",
	0x03c20066 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_7__A",
	0x03c20067 => "OFDM_SC_RA_RAM_STATE_PROC_STOP_8__A",
	0x03c2006e => "OFDM_SC_RA_RAM_PILOT_POW_WEIGHT__A",
	0x03c2006f => "OFDM_SC_RA_RAM_PILOT_POW_TARGET__A",
	0x03c20070 => "OFDM_SC_RA_RAM_STATE_PROC_START_1__A",
	0x03c20071 => "OFDM_SC_RA_RAM_STATE_PROC_START_2__A",
	0x03c20072 => "OFDM_SC_RA_RAM_STATE_PROC_START_3__A",
	0x03c20073 => "OFDM_SC_RA_RAM_STATE_PROC_START_4__A",
	0x03c20074 => "OFDM_SC_RA_RAM_STATE_PROC_START_5__A",
	0x03c20075 => "OFDM_SC_RA_RAM_STATE_PROC_START_6__A",
	0x03c20076 => "OFDM_SC_RA_RAM_STATE_PROC_START_7__A",
	0x03c20077 => "OFDM_SC_RA_RAM_STATE_PROC_START_8__A",
	0x03c2007c => "OFDM_SC_RA_RAM_FR_THRES_2K__A",
	0x03c2007d => "OFDM_SC_RA_RAM_FR_THRES_8K__A",
	0x03c2007e => "OFDM_SC_RA_RAM_STATUS__A",
	0x03c2007f => "OFDM_SC_RA_RAM_NF_BORDER_INIT__A",
	0x03c20080 => "OFDM_SC_RA_RAM_TIMER__A",
	0x03c20081 => "OFDM_SC_RA_RAM_FI_OFFSET__A",
	0x03c20082 => "OFDM_SC_RA_RAM_ECHO_GUARD__A",
	0x03c2008e => "OFDM_SC_RA_RAM_IF_SAVE_0__A",
	0x03c2008f => "OFDM_SC_RA_RAM_IF_SAVE_1__A",
	0x03c20098 => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_2K_32__A",
	0x03c20099 => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_2K_16__A",
	0x03c2009a => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_2K_8__A",
	0x03c2009b => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_2K_4__A",
	0x03c2009c => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_8K_32__A",
	0x03c2009d => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_8K_16__A",
	0x03c2009e => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_8K_8__A",
	0x03c2009f => "OFDM_SC_RA_RAM_DIVERSITY_DELAY_8K_4__A",
	0x03c200b2 => "OFDM_SC_RA_RAM_TD_REQ_SMB_CNT__A",
	0x03c200b7 => "OFDM_SC_RA_RAM_MG_VALID_THRES__A",
	0x03c200b8 => "OFDM_SC_RA_RAM_MG_MAX_DAT_THRES__A",
	0x03c200b9 => "OFDM_SC_RA_RAM_MG_CORR_TIMEOUT_8K__A",
	0x03c200ba => "OFDM_SC_RA_RAM_PILOT_CPD_EXP_MARG_VAL__A",
	0x03c200bb => "OFDM_SC_RA_RAM_PILOT_CPD_EXP_MARG_COUNT__A",
	0x03c200bc => "OFDM_SC_RA_RAM_PILOT_SPD_THRES__A",
	0x03c200bd => "OFDM_SC_RA_RAM_PILOT_SPD_TIMEOUT__A",
	0x03c200be => "OFDM_SC_RA_RAM_PILOT_CPD_THRES__A",
	0x03c200bf => "OFDM_SC_RA_RAM_PILOT_CPD_TIMEOUT__A",
	0x03c200d0 => "OFDM_SC_RA_RAM_IR_FREQ__A",
	0x03c200d1 => "OFDM_SC_RA_RAM_IR_COARSE_2K_LENGTH__A",
	0x03c200d2 => "OFDM_SC_RA_RAM_IR_COARSE_2K_FREQINC__A",
	0x03c200d3 => "OFDM_SC_RA_RAM_IR_COARSE_2K_KAISINC__A",
	0x03c200d4 => "OFDM_SC_RA_RAM_IR_COARSE_8K_LENGTH__A",
	0x03c200d5 => "OFDM_SC_RA_RAM_IR_COARSE_8K_FREQINC__A",
	0x03c200d6 => "OFDM_SC_RA_RAM_IR_COARSE_8K_KAISINC__A",
	0x03c200d7 => "OFDM_SC_RA_RAM_IR_FINE_2K_LENGTH__A",
	0x03c200d8 => "OFDM_SC_RA_RAM_IR_FINE_2K_FREQINC__A",
	0x03c200d9 => "OFDM_SC_RA_RAM_IR_FINE_2K_KAISINC__A",
	0x03c200da => "OFDM_SC_RA_RAM_IR_FINE_8K_LENGTH__A",
	0x03c200db => "OFDM_SC_RA_RAM_IR_FINE_8K_FREQINC__A",
	0x03c200dc => "OFDM_SC_RA_RAM_IR_FINE_8K_KAISINC__A",
	0x03c200dd => "OFDM_SC_RA_RAM_ECHO_SHIFT_LIM__A",
	0x03c200de => "OFDM_SC_RA_RAM_ECHO_SHT_LIM__A",
	0x03c200df => "OFDM_SC_RA_RAM_ECHO_SHIFT_TERM__A",
	0x03c200e0 => "OFDM_SC_RA_RAM_NI_INIT_2K_PER_LEFT__A",
	0x03c200e1 => "OFDM_SC_RA_RAM_NI_INIT_2K_PER_RIGHT__A",
	0x03c200e2 => "OFDM_SC_RA_RAM_NI_INIT_2K_POS_LR__A",
	0x03c200e3 => "OFDM_SC_RA_RAM_NI_INIT_8K_PER_LEFT__A",
	0x03c200e4 => "OFDM_SC_RA_RAM_NI_INIT_8K_PER_RIGHT__A",
	0x03c200e5 => "OFDM_SC_RA_RAM_NI_INIT_8K_POS_LR__A",
	0x03c200e7 => "OFDM_SC_RA_RAM_FREQ_OFFSET_LIM__A",
	0x03c200e8 => "OFDM_SC_RA_RAM_SAMPLE_RATE_COUNT__A",
	0x03c200e9 => "OFDM_SC_RA_RAM_SAMPLE_RATE_STEP__A",
	0x03c200ea => "OFDM_SC_RA_RAM_TPS_TIMEOUT_LIM__A",
	0x03c200eb => "OFDM_SC_RA_RAM_TPS_TIMEOUT__A",
	0x03c200ec => "OFDM_SC_RA_RAM_BAND__A",
	0x03c200ed => "OFDM_SC_RA_RAM_EC_OC_CRA_HIP_INIT__A",
	0x03c200ee => "OFDM_SC_RA_RAM_NE_ERR_SELECT_2K__A",
	0x03c200ef => "OFDM_SC_RA_RAM_NE_ERR_SELECT_8K__A",
	0x03c200f0 => "OFDM_SC_RA_RAM_REG_0__A",
	0x03c200f1 => "OFDM_SC_RA_RAM_REG_1__A",
	0x03c200f2 => "OFDM_SC_RA_RAM_BREAK__A",
	0x03c200f3 => "OFDM_SC_RA_RAM_BOOTCOUNT__A",
	0x03c200f4 => "OFDM_SC_RA_RAM_LC_ABS_2K__A",
	0x03c200f5 => "OFDM_SC_RA_RAM_LC_ABS_8K__A",
	0x03c200f6 => "OFDM_SC_RA_RAM_NE_NOTCH_WIDTH__A",
	0x03c200f7 => "OFDM_SC_RA_RAM_CP_GAIN_PEXP_SUB__A",
	0x03c200f8 => "OFDM_SC_RA_RAM_SRMM_FIX_FACT_8K__A",
	0x03c200f9 => "OFDM_SC_RA_RAM_LC_CP__A",
	0x03c200fa => "OFDM_SC_RA_RAM_LC_DIFF__A",
	0x03c200fb => "OFDM_SC_RA_RAM_ECHO_NF_THRES__A",
	0x03c200fc => "OFDM_SC_RA_RAM_ECHO_NF_FEC__A",
	0x03c200fd => "OFDM_SC_RA_RAM_ECHO_RANGE_OFS__A",
	0x03c200fe => "OFDM_SC_RA_RAM_RELOCK__A",
	0x03c200ff => "OFDM_SC_RA_RAM_STACKUNDERFLOW__A",
	0x03c20148 => "OFDM_SC_RA_RAM_NF_MAXECHOTOKEN__A",
	0x03c20149 => "OFDM_SC_RA_RAM_NF_PREPOST__A",
	0x03c2014a => "OFDM_SC_RA_RAM_NF_PREBORDER__A",
	0x03c2014b => "OFDM_SC_RA_RAM_NF_START__A",
	0x03c2014c => "OFDM_SC_RA_RAM_NF_MINISI_0__A",
	0x03c2014d => "OFDM_SC_RA_RAM_NF_MINISI_1__A",
	0x03c2014f => "OFDM_SC_RA_RAM_NF_NRECHOES__A",
	0x03c20150 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_0__A",
	0x03c20151 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_1__A",
	0x03c20152 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_2__A",
	0x03c20153 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_3__A",
	0x03c20154 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_4__A",
	0x03c20155 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_5__A",
	0x03c20156 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_6__A",
	0x03c20157 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_7__A",
	0x03c20158 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_8__A",
	0x03c20159 => "OFDM_SC_RA_RAM_NF_ECHOTABLE_9__A",
	0x03c2015a => "OFDM_SC_RA_RAM_NF_ECHOTABLE_10__A",
	0x03c2015b => "OFDM_SC_RA_RAM_NF_ECHOTABLE_11__A",
	0x03c2015c => "OFDM_SC_RA_RAM_NF_ECHOTABLE_12__A",
	0x03c2015d => "OFDM_SC_RA_RAM_NF_ECHOTABLE_13__A",
	0x03c2015e => "OFDM_SC_RA_RAM_NF_ECHOTABLE_14__A",
	0x03c2015f => "OFDM_SC_RA_RAM_NF_ECHOTABLE_15__A",
	0x03c201a0 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_UNKNOWN_MAN__A",
	0x03c201a1 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_UNKNOWN_EXP__A",
	0x03c201a2 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_QPSK_MAN__A",
	0x03c201a3 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_QPSK_EXP__A",
	0x03c201a4 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_16QAM_MAN__A",
	0x03c201a5 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_16QAM_EXP__A",
	0x03c201a6 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_16QAM_A2_MAN__A",
	0x03c201a7 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_16QAM_A2_EXP__A",
	0x03c201a8 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_16QAM_A4_MAN__A",
	0x03c201a9 => "OFDM_SC_RA_RAM_EQ_IS_GAIN_16QAM_A4_EXP__A",
	0x03c201aa => "OFDM_SC_RA_RAM_EQ_IS_GAIN_64QAM_MAN__A",
	0x03c201ab => "OFDM_SC_RA_RAM_EQ_IS_GAIN_64QAM_EXP__A",
	0x03c201ac => "OFDM_SC_RA_RAM_EQ_IS_GAIN_64QAM_A2_MAN__A",
	0x03c201ad => "OFDM_SC_RA_RAM_EQ_IS_GAIN_64QAM_A2_EXP__A",
	0x03c201ae => "OFDM_SC_RA_RAM_EQ_IS_GAIN_64QAM_A4_MAN__A",
	0x03c201af => "OFDM_SC_RA_RAM_EQ_IS_GAIN_64QAM_A4_EXP__A",
	0x03c201fe => "OFDM_SC_RA_RAM_DRIVER_VERSION_0__A",
	0x03c201ff => "OFDM_SC_RA_RAM_DRIVER_VERSION_1__A",
	0x03c30000 => "OFDM_SC_IF_RAM_TRP_RST_0__A",
	0x03c30001 => "OFDM_SC_IF_RAM_TRP_RST_1__A",
	0x03c30002 => "OFDM_SC_IF_RAM_TRP_BPT0_0__A",
	0x03c30004 => "OFDM_SC_IF_RAM_TRP_BPT0_1__A",
	0x03c30004 => "OFDM_SC_IF_RAM_TRP_STKU_0__A",
	0x03c30005 => "OFDM_SC_IF_RAM_TRP_STKU_1__A",
	0x03c30ffe => "OFDM_SC_IF_RAM_VERSION_MA_MI__A",
	0x03c30fff => "OFDM_SC_IF_RAM_VERSION_PATCH__A",
);

my $old_flags = -1;
my $old_reg;
my $old_timestamp;
sub parse_drxk_addr($$$$$)
{
	my $timestamp = shift;
	my $addr = shift;
	my $n = shift;
	my $app_data = shift;
	my $write = shift;
	my $j = 0;

	my ($reg, $flags, $cmd, $bits);

	goto parse_error if ($n < 2);

	if ($old_flags < 0) {
		my $r0 = hex(substr($app_data, $j, 2));
		my $r1 = hex(substr($app_data, $j + 3, 2));
		$n -= 2;
		$j += 6;

		if ($r0 & 1) {
			goto parse_error if ($n < 2);
			# Decode a Long format
			my $r2 = hex(substr($app_data, $j, 2));
			my $r3 = hex(substr($app_data, $j + 3, 2));
			$n -= 2;
			$j += 6;

			$reg = ($r0 >> 1) |
			       ($r1 << 16) |
			       (($r2 & 0x0f) << 24) |
			       ($r3 << 7);

			$flags = $r2 & 0xf0;
		} else {
			$flags = 0;
			$reg = ($r0 >> 1) |
			       ($r1 & 0x0f << 16) |
			       ($r1 & 0xf0 << 18);
		}
	} else {
		goto parse_error if ($write);	# Parse error!!! Should be a read!

		$flags = $old_flags;
		$reg = $old_reg;
		$old_flags = -1;
	}

	if ($n == 0) {
		goto parse_error if (!$write);	# Parse error!!! Should be a write!

		$old_flags = $flags;
		$old_reg = $reg;
		$old_timestamp = $timestamp;
		return;
	}

	# Hack: may be useful for firmware dumps
	goto parse_block if ($n > 1 && $force_wrt_blocks);

	goto parse_block if ($n > 4 || $n == 3);
	goto parse_error if ($n != 2 && $n != 4);

	if (defined($drxk_map{$reg})) {
		$reg = $drxk_map{$reg};
	} else {
		$reg = sprintf "0x%08x", $reg;
	}

	my $data;
	if ($n == 2) {
		$bits = 16;
		$data = hex(substr($app_data, $j, 2)) |
			hex(substr($app_data, $j + 3, 2)) << 8;
	} else {
		$bits = 32;
		$data = hex(substr($app_data, $j, 2)) |
			hex(substr($app_data, $j + 3, 2)) << 16 |
			hex(substr($app_data, $j + 6, 2)) << 24 |
			hex(substr($app_data, $j + 9, 2)) << 8 if ($n > 2);
	}

	if ($flags) {
		my $descr;

		# That _seems_ to be the flags. Not sure through
		$descr .= "R/W/Modify " if ($flags & 0x10);
		$descr .= "Broadcast " if ($flags & 0x20);
		$descr .= "SingleMaster " if (($flags & 0xc0) == 0xc0);
		$descr .= "MultiMaster " if (($flags & 0xc0) == 0x40);
		$descr .= "ClearCRC " if (($flags & 0xc0) == 0x80);

		printf "$timestamp " if ($show_timestamp);

		if ($write) {
			printf "drxk_write%d_flags(state, 0x%02x, %s, 0x%08x, 0x%02x); /* Flags = %s */\n", $bits, $addr, $reg, $data, $flags, $descr;
		} else {
			printf "drxk_read%d_flags(state, 0x%02x, %s, 0x%02x) /* Read = 0x%08x Flags = %s */\n", $bits, $addr, $reg, $flags, $data, $descr;
		}

	} else {
		printf "$timestamp " if ($show_timestamp);
		if ($write) {
			printf "drxk_write%d(state, 0x%s, 0x%02x, 0x%08x, 0x%08x);\n", $bits, $addr, $reg, $data;
		} else {
			printf "drxk_read%d(state, 0x%s, 0x%02x, 0x%08x); /* Read = 0x%08x */\n", $bits, $addr, $reg, $data;
		}
	}

	return;

parse_block:
	# On several cases, we don't want to get the reg name
	# (e. g. during microcode load)
	$reg = sprintf "0x%08x", $reg;

	my $data = add_hex_mark(substr($app_data, $j));

	if ($flags) {
		my $descr;

		# That _seems_ to be the flags. Not sure through
		$descr .= "R/W/Modify " if ($flags & 0x10);
		$descr .= "Broadcast " if ($flags & 0x20);
		$descr .= "SingleMaster " if (($flags & 0xc0) == 0xc0);
		$descr .= "MultiMaster " if (($flags & 0xc0) == 0x40);
		$descr .= "ClearCRC " if (($flags & 0xc0) == 0x80);

		printf "$timestamp " if ($show_timestamp);
		if ($write) {
			printf "drxk_write_block_flags(state, 0x%02x, %s, %d, %s, %d); /* Flags = %s */\n", $addr, $reg, $n, $data, $flags, $descr;
		} else {
			printf "drxk_read_block_flags(state, 0x%02x, %s, %d, %d); /* Read = %s Flags = %s */\n", $addr, $reg, $n, $flags, $data, $descr;
		}

	} else {
		printf "$timestamp " if ($show_timestamp);
		if ($write) {
			printf "drxk_write_block(state, 0x%02x, %s, %d, %s);\n", $addr, $reg, $n, $data;
		} else {
			printf "drxk_read_block(state, 0x%02x, %s, %d); /* Read = %s */\n", $addr, $reg, $n, $data;
		}
	}

	return;

parse_error:
	# Fallback: Couldn't parse it
	if ($write) {
		if ($old_flags > 0) {
			if (defined($drxk_map{$old_reg})) {
				$old_reg = $drxk_map{$old_reg};
			} else {
				$old_reg = sprintf "0x%08x", $old_reg;
			}
			printf "$old_timestamp " if ($old_timestamp);
			printf "ERR: DRX-K write(state, 0x%02x, %s, 0x%08x) without data. Probably an read ops + read error\n", $bits, $addr, $old_reg, $old_flags;
		}
		printf "$timestamp " if ($show_timestamp);
		my $data = add_hex_mark($app_data);
		printf "i2c_master_send(0x%02x >> 1, %s, %d);\n", $addr, $data, $n;
	} else {
		printf "$timestamp " if ($show_timestamp);
		my $data = add_hex_mark($app_data);
		printf "i2c_master_recv(0x%02x >> 1, &buf, %d); /* %s */\n", $addr, $n, $data;
	}
}

sub i2c_decode($$$$$)
{
	my $reqtype = shift;
	my $wvalue  = shift;
	my $windex  = shift;
	my $wlen    = shift;
	my $payload = shift;

	my ($data, $addr, $write, $n);

	$n = $wlen;

	$write = 1 if (!($reqtype & 0x80));

	if ($n >= 6 && !$write) {
		# This is how az6007 returns reads
		$n -= 6;
		$payload = substr($payload, 3 * 5, $n * 3);
	}

	$addr = $wvalue & 0xff;

	if ($wvalue > 255) {
		$data = sprintf("%02x %s", $windex, $payload);
		$n++;
	} else {
		$data = sprintf("%s", $payload);
	}


	return ($addr, $data, $write, $n);
}

# This is a guess map for the Cypress FX2 registers that are seen
# at the device and/or at other devices with FX2. It is incomplete
# and may be wrong. The names here don't match the az6007 driver yet,
# but are useful for someone that is checking the logs to analyze
# what's wrong there
my %req_map = (
	# FX2 Special registers defined at CY7 datasheets
	0x80 => "FX2_IOA",
	0x90 => "FX2_IOB",
	0xa0 => "FX2_LOAD_FIRMWARE",
	0xb0 => "FX2_IOD",
	0xb5 => "FX2_OED",
	0xc0 => "FX2_SCON1",		# Used to reset demod
	0xd0 => "FX2_PSW",

	# Other registers used at the driver
	0xb7 => "AZ6007_READ_DATA", # I suspect that this is a RAM read function
	0xb9 => "AZ6007_I2C_RD",
	0xbc => "AZ6007_POWER",
	0xbd => "AZ6007_I2C_WR",	# FX2 calls it as GPIFSGL-DATH
	0xb4 => "AZ6007_READ_IR",
	0xc5 => "AZ6007_CI_STATUS",	# Guessed from AZ6027 driver
	0xc7 => "AZ6007_TS_THROUGH",
);

##############
# Main program
##############

while (<>) {
	my $org_line = $_;

	if (m/(.*)([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].)\s*([\<\>]+)\s*(.*)/) {
		my $timestamp = $1;
		my $reqtype = hex($2);
		my $req = hex($3);
		my $wvalue = hex("$5$4");
		my $windex = hex("$7$6");
		my $wlen = hex("$9$8");
		my $dir = $10;
		my $payload = $11;

		if ($req == 0xb9 || $req == 0xbd) {
			my ($addr, $data, $write, $n) = i2c_decode($reqtype, $wvalue, $windex, $wlen, $payload);

			if ($addr == 0x52) {
				parse_drxk_addr($timestamp, $addr, $n, $data, $write) if ($show_drxk);
			} elsif ($addr == 0xc0) {
				parse_mt2063_addr($timestamp, $addr, $n, $data, $write) if ($show_mt2063);
			} elsif ($show_other_xfer) {
				printf "$timestamp " if ($show_timestamp);
				if ($write) {
					printf "i2c_master_send(0x%02x >> 1, %s, %d);\n",
						$addr, add_hex_mark($data), $n;
				} else {
					printf "i2c_master_recv(0x%02x >> 1, &buf, %d); /* %s */\n",
						$addr, $n, add_hex_mark($data);
				}
			}
			next;
		}
		if ($show_other_reqs) {
			printf "$timestamp " if ($show_timestamp);

			if (defined($req_map{$req})) {
				$req = sprintf "%s", $req_map{$req}, $req;
			} else {
				$req = sprintf "0x%02x", $req;
			}

			if ($reqtype > 0x80) {
				printf("az6007_read(udev, %s, %s, %s, &data, %s); /* %s */\n",
					$req, $wvalue, $windex, $wlen, add_hex_mark($payload));
			} else {
				printf("az6007_write(udev, %s, %s, %s, %s, %s);\n",
					$req, $wvalue, $windex, add_hex_mark($payload), $wlen);
			}
		}
		next;
	}
	print $org_line if ($show_other_lines);
}
